<?php

namespace App\Http\Controllers\Branch;

use App\Http\Controllers\Controller;
use App\Models\Branch\Appointment;
use App\Models\Branch\Branch;
use App\Models\Branch\Lab;
use App\Models\Invoice\Invoice;
use App\Models\Invoice\Invoice_item;
use App\Models\Patient\Service_item;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use PDF;
use Image;
use Mpdf\Mpdf;

class LabController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $branches = Branch::select('id', 'name')->get();
        return view('lab.index', compact('branches'));
    }

    public function lab_result_nopay_index($branch)
    {
        $branch = request()->branch;

        $lab = Lab::select('id', 'services_cat_id', 'patient_id', 'created_at')
            ->Where('branch_id', $branch)
            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['patient' => function ($q) {
                $q->select('id', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['invoice_item' => function ($q) {
                $q->select('id', 'invoice_id', 'itemable_id', 'itemable_type')
                    ->with(['invoice' => function ($q) {
                        $q->select('id', 'code', 'status');
                    }]);
            }])
            ->whereHas('invoice_item.invoice', function ($query) {
                return $query->where('status', '0');
            })
            ->orderBy('status', 'ASC')
            ->limit(10)
            ->get();

        return $lab;
    }


    public function lab_result_index($branch)
    {
        $branch = request()->branch;

        $lab = Lab::select('id', 'services_cat_id', 'patient_id', 'created_at', 'status')
            ->Where('branch_id', $branch)
            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['patient' => function ($q) {
                $q->select('id', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['invoice_item' => function ($q) {
                $q->select('id', 'invoice_id', 'itemable_id', 'itemable_type')
                    ->with(['invoice' => function ($q) {
                        $q->select('id', 'code', 'status');
                    }]);
            }])
            ->whereHas('invoice_item.invoice', function ($query) {
                return $query->where('status', '2');
            })
            ->orderBy('status', 'ASC')
            ->limit(10)
            ->get();

        return $lab;
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
    }

    public function patient_add(Request $request)
    {

        $this->validate($request, [
            'lab_cat_serv_id' => ['required', 'exists:service_items,id'],
            'last_appointment_id' => ['required', 'exists:appointments,id'],
            'patient_id' => ['required', 'exists:patients,id'],
        ]);

        $lab_cat_serv_id = $request->input('lab_cat_serv_id');

        $branch_id = Appointment::select('branch_id')->find($request->input('last_appointment_id'));

        $service_price = Service_item::select('id', 'service_inv_cat_id', 'price')->where('id', $lab_cat_serv_id)->where('deactivate', 0)->first();

        $lab = Lab::create([
            'code' => "LB" . $this->generateRandomString(6),
            'services_cat_id' => $lab_cat_serv_id,
            'appointment_id' => $request->input('last_appointment_id'),
            'patient_id' => $request->input('patient_id'),
            'branch_id' => $branch_id->branch_id,
            'note_doctor' => $request->input('note_doctor_lab'),
        ]);

        //for new invoice
        if ($request->input('lab_cat_invoice') === 'new') {
            $invoice = Invoice::create([
                'code' => "IN" . $this->generateRandomString(6),
                'service_inv_cat_id' => $service_price->service_inv_cat_id,
                'specialty_id' => Auth::user()->specialty_id,
                'branch_id' => $branch_id->branch_id,
                'receivable_id' => $request->input('patient_id'),
                'receivable_type' => "App\Models\Patient\Patient",
                'items_price' => $service_price->price,
                'final_price' => $service_price->price,
                'note' => $request->input('invoice_note_lab'),
            ]);
            $invoice_item = Invoice_item::create([
                'invoice_id' => $invoice->id,
                'itemable_id' => $lab->id,
                'itemable_type' => "App\Models\Branch\Lab",
                'categorizable_id' => $lab_cat_serv_id,
                'categorizable_type' => "App\Models\Patient\Service_item",
                'price' => $service_price->price,
                'sold_price' => $service_price->price,
            ]);
        } else {
            $invoice_item = Invoice_item::create([
                'invoice_id' => $request->input('lab_cat_invoice'),
                'itemable_id' => $lab->id,
                'itemable_type' => "App\Models\Branch\Lab",
                'categorizable_id' => $lab_cat_serv_id,
                'categorizable_type' => "App\Models\Patient\Service_item",
                'price' => $service_price->price,
                'sold_price' => $service_price->price,
            ]);

            $invoice = Invoice::find($request->input('lab_cat_invoice'));
            $invoice->increment('items_price', $service_price->price);
            $invoice->increment('final_price', $service_price->price);
            $invoice->note = $invoice->note . ' ' . $request->input('invoice_note_lab');
            $invoice->save();
        }

        session()->flash('success', 'The X-RAY has been created successfully');
        return redirect()->back();
    }

    public function generateRandomString($length = 20)
    {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $lab = Lab::select('id', 'code', 'services_cat_id', 'appointment_id', 'patient_id', 'branch_id', 'resp_doctor_id', 'xray_file', 'note_doctor', 'note_lab', 'status', 'created_at')
            ->Where('id', $id)
            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['branch' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['patient' => function ($q) {
                $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['invoice_item' => function ($q) {
                $q->select('id', 'invoice_id', 'itemable_id', 'itemable_type')
                    ->with(['invoice' => function ($q) {
                        $q->select('id', 'code', 'status');
                    }]);
            }])
            ->with(['appointment' => function ($q) {
                $q->select('id', 'start_at');
            }])
            ->with(['doctor' => function ($q) {
                $q->select('id', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->orderBy('status', 'ASC')
            ->first();

        return view('lab.show', compact('lab'));
    }

    //select the searched patients via ajax
    public function lab_search($search_query)
    {
        $search_query = request()->search_query;

        $lab = Lab::with(['patient' => function ($q) {
            $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
        }])
            ->select('id', 'code', 'patient_id', 'created_at')
            ->Where('code', 'like', "%{$search_query}%")
            ->orWhereHas('patient', function ($query) use ($search_query) {
                $query->whereRaw("concat(first_name, ' ', second_name) like '%$search_query%' ");
            })
            ->limit(5)
            ->get();

        return $lab;
    }



    //print 
    public function lab_print($id)
    {
        $lab = Lab::select('id', 'code', 'services_cat_id', 'appointment_id', 'patient_id', 'branch_id', 'resp_doctor_id', 'xray_file', 'note_doctor', 'note_lab', 'status', 'created_at')
            ->Where('id', $id)
            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['branch' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['patient' => function ($q) {
                $q->select('id', 'avatar', 'username', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['appointment' => function ($q) {
                $q->select('id', 'start_at');
            }])
            ->with(['doctor' => function ($q) {
                $q->select('id', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->orderBy('status', 'ASC')
            ->first();

        $qrcode = base64_encode(QrCode::color(68, 95, 129)->size(80)->style('round')->eye('circle')->generate($lab->code));

        // instantiate and use the dompdf class

        $html = view('lab.print', compact('lab', 'qrcode'))->render();

        $mpdf = new Mpdf();

        $mpdf->SetHTMLHeader('<div class="header">
        <div class="row">
        <div class="col-xs-9 m-0 ps-3">
            <img src="' . public_path('img/dashboard/system/pc-loader.png') . '">
        </div>
        <div class="col-xs-2 text-center m-0 ps-3">
            <img src="data:image/svg;base64,' . $qrcode . '">
            <p class="mt-1 mb-0 text-s fw-bold2">' . $lab->code . '</p>
        </div>
        </div>
        </div>
        ', 'O');
        $mpdf->SetHTMLFooter('<div class"text-center"> Page {PAGENO} <span class="text-s">Copyright © 2021 Proxima, Developed
        by SHM/span> | {DATE j-m-Y}</div>');

        $mpdf->AddPageByArray([
            'margin-top' => 43,
            'margin-bottom' => 17,
        ]);

        $mpdf->WriteHTML($html);
        $mpdf->output();
    }



    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {

        $lab = Lab::select('id', 'code', 'services_cat_id', 'appointment_id', 'patient_id', 'resp_doctor_id', 'xray_file', 'note_doctor', 'note_lab', 'status', 'created_at')
            ->Where('id', $id)
            ->with(['service_item' => function ($q) {
                $q->select('id', 'name');
            }])
            ->with(['patient' => function ($q) {
                $q->select('id', 'avatar', DB::raw('CONCAT(first_Name, " ", second_Name) AS name'));
            }])
            ->with(['invoice_item' => function ($q) {
                $q->select('id', 'invoice_id', 'itemable_id', 'itemable_type')
                    ->with(['invoice' => function ($q) {
                        $q->select('id', 'code', 'status');
                    }]);
            }])
            ->with(['appointment' => function ($q) {
                $q->select('id', 'start_at');
            }])
            ->orderBy('status', 'ASC')
            ->first();

        return view('lab.edit', compact('lab'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'responsible_doctor' => ['required', 'exists:users,id'],
            'x_ray_file' => ['image', 'mimes:jpeg,jpg,png', 'max:1000'],
        ]);

        $lab = Lab::find($id);

        $xray_file = $request->file('x_ray_file');

        if ($request->hasFile('x_ray_file')) {

            //to remove the old avatar and also keep the default img
            $imagePath = public_path('img/lab/' . $lab->xray_file);
            if (File::exists($imagePath)) {
                File::delete($imagePath);
            }

            $file_extension = $request->file('x_ray_file')->getClientOriginalExtension();

            $file_name = "LA" . time() . '.' . $file_extension;

            $path = public_path('img/lab/' . $file_name);

            Image::make($request->x_ray_file)
                ->resize(500, null, function ($constraint) {
                    $constraint->aspectRatio();
                })
                ->save($path, 60);

            $lab->xray_file = $file_name; //new img file name to upload to database
            $lab->status = "1";
        }

        $lab->resp_doctor_id = $request->input('responsible_doctor');
        $lab->note_lab = $request->input('lab_note');

        $lab->save();

        session()->flash('success', 'The lab result has been updated');
        return redirect()->route('sett.lab.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $this->validate($request, [
            'lab_invoice_id_delete' => ['required', 'invoice_delete'],
        ]);

        $id = $request->input('lab_id_delete');
        $invoice_id = $request->input('lab_invoice_id_delete');

        Lab::find($id)->delete();
        Invoice::find($invoice_id)->delete();

        session()->flash('success', 'The X-Ray has been deleted');
        return redirect()->back();
    }
}
