<?php

namespace App\Services\Exams;

use App\Http\Resources\ManagerDashboard\Exams\StudentCommittes\CommitteeDataResource;
use App\Http\Resources\ManagerDashboard\Exams\StudentCommittes\StudentCommitteeResource;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\SchoolSemester;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\Exams\StudentCommittees\StudentCommittee;
use App\Models\ManagerDashboard\Exams\StudentCommittees\StudentCommitteeData;
use App\Models\Namazeg;
use App\Traits\NamazegTrait;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class CommitteeService
{

    use  NamazegTrait;

    public function createAutomaticCommittees(array $data, $school)
    {

        return  DB::transaction(function () use ($data, $school) {

            $namozeg = Namazeg::where('form_number', 61)->first();

            if (!$namozeg) {
                throw new \Exception('Namozeg not found for this committee form number in Namazeg table.');
            }

            // Get or create Namazeg version
            $namozeg_version = $this->storeNamazegVersion(
                $namozeg->id,
                $namozeg->internal_extenral,
                $school->id
            );

            // Find SchoolClass
            $schoolClass = SchoolClass::findOrFail($data['schoolclass_id']);



            // التأكد من أن هذا الفصل لم يتم تشكيل لجنة له سابقًا
            if ($schoolClass->studentCommittees()
                ->where('semester_id', $data['semester_id'])
                ->where('school_id', $school->id)
                ->exists()
            ) {
                throw new \Exception('تم تشكيل لجنة لهذا الفصل مسبقًا');
            }

            $committeeType = ($schoolClass->entsab === 'yes') ? 'entsab' : 'entezam';


            // Get the next committee number
            $committeeNumber = $this->getNextCommitteeNumber($school->id, $data['semester_id'], $committeeType);
            // Determine location if not provided

            if ($committeeType === 'entsab') {
                if (!isset($data['location'])) {
                    throw new \Exception('يجب اختيار مقر للجنة الانتساب');
                }
            } else {
                $data['location'] = $schoolClass->className->name . $schoolClass->class_number;
            }

            $location = $data['location'];

            // Create and store committee
            $committee = StudentCommittee::create([
                'namazeg_version_id'   => $namozeg_version->id,
                'school_id'            => $school->id,
                'semester_id'          => $data['semester_id'],
                'schoolclass_id'       => $data['schoolclass_id'],
                'name'                 => "اللجنة " . $committeeNumber,
                'committee_student_type' => $committeeType,
                'location'             =>  $location,
                'formation_type'       => 'automatic',
                'status' => 'complated'
            ]);


            $students = $schoolClass->Students()->where('school_id', $school->id)->get();

            foreach ($students as $student) {

                StudentCommitteeData::create([
                    'student_committee' =>  $committee->id,
                    'student_id' =>  $student->id,

                ]);
            }


            return $committee;
        });
    }

    /**
     * Create manual committees.
     */
    public function createManualCommittees(array $data, $school)
    {

        DB::transaction(function () use ($data, $school) {
            // Find Namozeg
            $namozeg = Namazeg::where('form_number', 61)->first();
            if (!$namozeg) {
                throw ValidationException::withMessages(['error' => 'Namozeg not found for this committee form number in Namazeg table.']);
            }

            // If no version exists, create a new one using the trait method
            $namozeg_version = $this->storeNamazegVersion(
                $namozeg->id,
                $namozeg->internal_extenral,
                $school->id
            );

            // Create multiple committees
            for ($i = 0; $i < $data['commitees_number']; $i++) {

                $committeeNumber = $this->getNextCommitteeNumber($school->id, $data['semester_id'], $data['committee_student_type']);

                StudentCommittee::create([
                    'namazeg_version_id' => $namozeg_version->id,
                    'school_id' => $school->id,
                    'semester_id' => $data['semester_id'],
                    'name' => "اللجنة " . $committeeNumber,
                    'committee_student_type' => $data['committee_student_type'],
                    'formation_type' => 'manual',
                ]);
            }

            return true;
        });
    }

    public function assignManualCommitteeStudents($data, $committee)
    {
        DB::beginTransaction(); // Start transaction

        try {
            // Update committee location and status
            $committee->update(['location' => $data['location'], 'status' => 'complated']);

            // Remove previously assigned students in the committee
            $committee->committeeStudents()->delete();

            $students = [];
            foreach ($data['assignments'] as $assignment) {
                foreach ($assignment['student_ids'] as $studentID) {
                    $students[] = [
                        'student_committee' => $committee->id,
                        'student_id' => $studentID,
                        'column_number' => $assignment['column_number'],
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }
            }

            // Batch insert students if the array is not empty

            StudentCommitteeData::insert($students);

            DB::commit(); // Commit transaction

            return true;
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback transaction if an error occurs

            throw new \Exception('حدث خطأ أثناء تعيين الطلاب.');
        }
    }

    private function getNextCommitteeNumber($schoolId, $semesterId, $committeeType)
    {
        $lastCommittee = StudentCommittee::where('school_id', $schoolId)
            ->where('semester_id', $semesterId)
            ->where('committee_student_type', $committeeType)
            ->orderBy('id', 'desc')
            ->first();

        if ($lastCommittee && preg_match('/\d+$/', $lastCommittee->name, $matches)) {
            return intval($matches[0]) + 1;
        }

        return 1; // Default to 1 if no previous committees exist
    }


    public function getAllCommittees($school)
    {
        $schoolSemesters = SchoolSemester::with([
            'exam_student_committees' => fn($query) => $query->where('school_id', $school->id)
        ])->where('status', '1')->get();

        $response = $schoolSemesters->map(function ($semester) {

            $groupedCommittees = $semester->exam_student_committees->groupBy('committee_student_type');

            $formattedCommittees = $groupedCommittees->mapWithKeys(function ($committees, $type) {
                return [
                    $type => [
                        'committees' => StudentCommitteeResource::collection($committees)
                    ]
                ];
            });

            return [
                'semester_id' => $semester->id,
                'semester_name' => $semester->name,
                'committees' => $formattedCommittees
            ];
        });

        return response()->json(['schoolSemesters' => $response]);
    }

    public function getCommitteeDetails($school, $committee)
    {

        $studentsIds = StudentCommitteeData::where('student_committee', $committee->id)
            ->pluck('student_id');

        $students = Student::whereIn('id', $studentsIds)->get();


        $class_names = Student::whereHas('studentCommitteeData', function ($query) use ($committee) {
            $query->where('student_committee', $committee->id);
        })
            ->join('school_classes', 'school_students.class_id', '=', 'school_classes.id') // Join to get class_name_id
            ->join('class_names', 'school_classes.class_name_id', '=', 'class_names.id') // Join to get class name
            ->selectRaw('class_names.id as class_name_id, class_names.name as class_name, COUNT(school_students.id) as students_count')
            ->groupBy('class_names.id', 'class_names.name') // Group by class_name_id and name
            ->get();

        $committee_plan = StudentCommitteeData::from('exams_student_committee_data as scd')
            ->where('student_committee', $committee->id)
            ->whereNotNull('column_number')
            ->select(
                'column_number',
                'class_names.name as class_name',
                'school_classes.class_number',
                DB::raw('COUNT(scd.student_id) as student_count')
            )
            ->join('school_students', 'scd.student_id', '=', 'school_students.id')
            ->join('school_classes', 'school_students.class_id', '=', 'school_classes.id')
            ->join('class_names', 'school_classes.class_name_id', '=', 'class_names.id')
            ->groupBy('column_number', 'class_names.name', 'school_classes.class_number')
            ->get()
            ->map(function ($item) use ($committee) {
                $students = Student::whereHas('studentCommitteeData', function ($query) use ($committee, $item) {
                    $query->where('student_committee', $committee->id)
                        ->where('column_number', $item->column_number);
                })->get();

                $item->students = $students;

                return $item;
            });


        return new CommitteeDataResource($committee, $students, $class_names, $committee_plan);
    }
    public function deleteCommittee($school, $committee)
    {

        if ($committee->formation_type == 'automatic') {
            return $committee->delete();
        } else {
            $committee->update([
                'location' => null,
                'status' => 'incomplated'
            ]);
            return $committee->committeeStudents()->delete();
        }
    }

    public function deleteAllCommittees($school, $semester)
    {

        return  StudentCommittee::where('school_id', $school->id)->where('semester_id', $semester->id)->forceDelete();
    }

    public function getRemainingStudentsSummary($school, $semester)
    {
        // Count the remaining incomplete committees for the school and semester
        $remainingCommittees = StudentCommittee::where('school_id', $school->id)
            ->where('semester_id', $semester->id)
            ->where('status', 'incomplated')
            ->count();


        $remaining_students = Student::where('school_students.school_id', $school->id)
            ->whereDoesntHave('studentCommitteeData', function ($query) use ($school, $semester) {
                $query->whereHas('studentCommittee', function ($subQuery) use ($school, $semester) {
                    $subQuery->where('exams_student_committees_table.school_id', $school->id)
                        ->where('exams_student_committees_table.semester_id', $semester->id);
                });
            })->count();

        $remaining_class_names_students = Student::where('school_students.school_id', $school->id)
            ->whereDoesntHave('studentCommitteeData', function ($query) use ($school, $semester) {
                $query->whereHas('studentCommittee', function ($subQuery) use ($school, $semester) {
                    $subQuery->where('exams_student_committees_table.school_id', $school->id)
                        ->where('exams_student_committees_table.semester_id', $semester->id);
                });
            })
            ->join('school_classes', 'school_students.class_id', '=', 'school_classes.id')
            ->join('class_names', 'school_classes.class_name_id', '=', 'class_names.id')
            ->where('class_names.school_grade_id', $school->school_grade_id)
            ->selectRaw('class_names.name as class_name, COUNT(school_students.id) as students_count')
            ->groupBy('class_names.name')
            ->get();

        return response()->json([
            'remainingCommittees' => $remainingCommittees,
            'remaining_students' => $remaining_students,
            'remainingStudents' => $remaining_class_names_students->map(function ($remaining_class_names_student) {
                return [
                    'class_name' => $remaining_class_names_student->class_name,
                    'remaining_students' => $remaining_class_names_student->students_count,
                ];
            }),
        ]);
    }


    public function duplicateSemesterCommittees($school, array $data)
    {

        DB::beginTransaction();

        try {

            $namozeg = Namazeg::where('form_number', 61)->first();

            if (!$namozeg) {
                throw new \Exception('Namozeg not found for this committee form number in Namazeg table.');
            }

            // Get or create Namazeg version
            $namozeg_version = $this->storeNamazegVersion(
                $namozeg->id,
                $namozeg->internal_extenral,
                $school->id
            );


            $previousSemesterId = $data['previous_semseter'];
            $currentSemesterId = $data['current_semester'];

            $previousCommittees = StudentCommittee::with('committeeStudents')
                ->where('school_id', $school->id)
                ->where('semester_id', $previousSemesterId)
                ->get();

            $existingCurrentCommittees = StudentCommittee::where('school_id', $school->id)
                ->where('semester_id', $currentSemesterId)
                ->exists();

            if ($existingCurrentCommittees) {
                throw new \Exception('هذا الفصل الدراسي يحتوي علي لجان بالفعل');
            }

            if ($previousCommittees->isEmpty()) {
                throw new \Exception('لا يوجد لجان في الفصل الدراسي السابق');
            }

            foreach ($previousCommittees as $previousCommittee) {
                // regenerate namazeg_version_id
                $previousCommittee['namazeg_version_id'] = $namozeg_version->id;
                $previousCommittee['formation_type'] = 'manual';
                $previousCommittee['status'] = 'incomplated';
                $newCommittee = $previousCommittee->replicate();
                $newCommittee->semester_id = $currentSemesterId;
                $newCommittee->save();

                // foreach ($previousCommittee->committeeStudents as $studentData) {
                //     $newData = $studentData->replicate();
                //     $newData->student_committee = $newCommittee->id;
                //     $newData->save();
                // }
            }

            DB::commit();
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            throw new \Exception($e->getMessage());
        }
    }
}
