<?php
namespace App\Services;

use Illuminate\Support\Facades\DB;
use App\Models\ManagerDashboard\Backup;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class DatabaseSwitcher
{
    private $mainDatabase;

    public function __construct()
    {
        $this->mainDatabase = env('DB_DATABASE');
    }

    private function generateTempDatabaseName($userId)
    {
        return "temp_preview_{$userId}_" . time();
    }

    private function createDatabase($databaseName)
    {
        // Use default connection to create database
        DB::statement("CREATE DATABASE IF NOT EXISTS `$databaseName`");
    }

    private function dropDatabase($databaseName)
    {
        if ($databaseName !== $this->mainDatabase) {
            // Use default connection to drop database
            DB::statement("DROP DATABASE IF EXISTS `$databaseName`");
        }
    }

    public function switchToArchive(Backup $backup)
    {
        try {
            $user = app('AppServiceProviderUser');
            $tempDatabase = $this->generateTempDatabaseName($user->id);
            
            // Create new temporary database
            $this->createDatabase($tempDatabase);
            
            // Create a new connection configuration for the temporary database
            $tempConfig = config('database.connections.mysql');
            $tempConfig['database'] = $tempDatabase;
            
            // Register the temporary connection
            config(['database.connections.temp_preview' => $tempConfig]);
            
            // Switch to temporary connection and import backup
            DB::purge('temp_preview');
            DB::reconnect('temp_preview');
            
            // Set as default connection
            DB::setDefaultConnection('temp_preview');
            
            // Import backup SQL to temporary database
            $backupFilePath = storage_path("app/backups/{$backup->name}.sql");
            
            if (!file_exists($backupFilePath)) {
                throw new \Exception('Backup file does not exist.');
            }
    
            $sql = file_get_contents($backupFilePath);
            
            if (!$sql) {
                throw new \Exception('Failed to read backup file.');
            }
    
            // Execute SQL import on temporary database
            DB::unprepared($sql);
            
            // Store the connection info in cache
            DatabaseContext::enableArchiveMode($tempDatabase);
    
            return true;
        } catch (\Exception $e) {
            Log::error("Archive switch error: " . $e->getMessage());
            throw $e;
        }
    }

    public function switchToMain()
    {
        try {
            $user = app('AppServiceProviderUser');
            
            // Get current preview database name before switching
            $previewUsers = Cache::get(DatabaseContext::PREVIEW_USERS_KEY, []);
            $tempDatabase = $previewUsers[$user->id]['temp_database'] ?? null;
            
            // Switch back to main connection
            DB::setDefaultConnection('mysql');
            DB::purge('mysql');
            DB::reconnect('mysql');
            
            // Remove temporary connection
            if ($tempDatabase) {
                // Drop the temporary database
                $this->dropDatabase($tempDatabase);
                config(['database.connections.temp_preview' => null]);
            }
            
            DatabaseContext::disableArchiveMode();

            return true;
        } catch (\Exception $e) {
            Log::error("Main database switch error: " . $e->getMessage());
            throw $e;
        }
    }
}