<?php
namespace App\Http\Controllers\Api\User;

use App\Http\Controllers\Controller;
use App\Traits\SmsTrait;
use Illuminate\Http\Request;
use JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;
use Illuminate\Support\Facades\Validator;
use Str;
use Hash;
use App\Models\User;
use DB;
use Spatie\Permission\Models\Role;
use App\Http\Resources\User\UserResource;
use App\Http\Resources\User\UserRoleResource;

class UserController extends Controller
{
    use SmsTrait;
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct(){
        //$this->middleware(['auth:api','permission:user-settings']);
    }

    /* function return userd profile informations */
    public function getUserProfile(){
        try {

            if (! $user = JWTAuth::parseToken()->authenticate()) {
                return response()->json(['user_not_found'], 404);
            }

        } catch (Tymon\JWTAuth\Exceptions\TokenExpiredException $e) {

            return response()->json(['token_expired'], $e->getStatusCode());

        } catch (Tymon\JWTAuth\Exceptions\TokenInvalidException $e) {

            return response()->json(['token_invalid'], $e->getStatusCode());

        } catch (Tymon\JWTAuth\Exceptions\JWTException $e) {

            return response()->json(['token_absent'], $e->getStatusCode());

        }
        $token = JWTAuth::fromUser($user);
        $user = new UserResource ($user);
        return response()->json(['userData'=>$user,'message'=>trans('api.user data')]);
    }

    /* funftion update user profile data */
    public function updateUserProfile(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'phone' => 'required|string|regex:/^05\d{8}$/|unique:users,phone,' . $user->id,
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors();
            foreach ($errors->all() as $message) {
                $error[] =  trans("api.$message");
            }
            return response()->json(['message' => trans('api.error entering data'), 'errors' => $validator->errors()], 400);
        }
        if ($user->phone != $request->phone) {
            $message_text = " تم تغيير رقم الجوال بنجاح.";
            static::sendSMSNotification($request->phone, $message_text, $user->manager_id);
        }

        $user->name=$request->name;
        $user->email=$request->email;
        $user->phone=$request->phone;
        $user->save();

        $token = JWTAuth::fromUser($user);
        $user = new UserResource ($user);

        return response()->json(['userData'=>$user,'message'=>trans('api.user data updated successfully')]);
    }

    /* function change user password */
    public function changePassword(Request $request){
        $validator = Validator::make($request->all(), [
            'password' => 'required|string|min:8',
        ]);

        if($validator->fails()){
            //return response()->json($validator->errors(), 400);

            $errors = $validator->errors();
            foreach ($errors->all() as $message) {
                $error[] =  trans("api.$message");
            }
            return response()->json(['message'=>trans('api.error entering data'),'errors'=>$validator->errors()],422);
        }

        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }

        if (Hash::check($request->old_password, $user->password) ) {
            $user->password = Hash::make($request->password);
            $user->password_reseted = 'yes';
            $user->save();
            return response()->json(['message'=>trans('api.password changed successfully')]);

        }else{
            return response()->json(['message'=>trans('api.old password is incorrect')],404);
        }

    }

    /** function return  all avialble roles*/
    public function getRoles(){
        $roles =UserRoleResource::collection(Role::get());
        return response()->json(['roles'=>$roles,'message'=>trans('api.list of all available records')]);
    }

    /** function change user account status*/
    public function changeStatus(Request $request){
        $user = User::find($request->account_id);
        if($user){
            /** update user account status */
            $user->status=$request->status;
            $user->save();
            return response()->json(['account_current_status'=>trans("api.$user->status"),'message'=>trans('api.account status changed')]);
        }else{
            return response()->json(['message'=>trans('api.no data found to this id')],404);
        }
    }

    /** function return all admin in the system*/
    public function getAllAdmins(){
        $admins =UserResource::collection(User::where('user_type','super_admin')->get());
        return response()->json(['admins'=>$admins,'message'=>trans('api.list of all available records')]);
    }

    /** create new admin and assign role to it */
    public function createNewAdmin(Request $request){
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'required|string|unique:users',
            'password' => 'required|string|min:8',
        ]);

        if($validator->fails()){
            //return response()->json($validator->errors(), 400);
            $errors = $validator->errors();
            foreach ($errors->all() as $message) {
                $error[] =  trans("api.$message");
            }
            return response()->json(['message'=>trans('api.error entering data'),'errors'=>$validator->errors()]);
        }

        $user = new User();
        $user->name = $request->name ;
        $user->email = $request->email ;
        $user->phone = $request->phone ;
        $user->password = Hash::make($request->password) ;
        $user->status = 'active' ;
        $user->user_type = 'super_admin' ;
        $user->verification_code = mt_rand(100000, 999999);
        $user->save();

        /** assign role to the new admin */
        $user->assignRole($request->role_name);

        $admin=new UserResource($user);

        return response()->json(['admin'=>$admin,'message'=>trans('api.new admin added successfully')]);
    }


    /** create new admin and assign role to it */
    public function updateAdmin(Request $request){
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'required|string|unique:users',
        ]);

        if($validator->fails()){
            //return response()->json($validator->errors(), 400);
            $errors = $validator->errors();
            foreach ($errors->all() as $message) {
                $error[] =  trans("api.$message");
            }
            return response()->json(['message'=>trans('api.error entering data'),'errors'=>$validator->errors()]);
        }

        $user = new User();
        $user->name = $request->name ;
        $user->email = $request->email ;
        $user->phone = $request->phone ;
        if($request->password){
            $user->password = Hash::make($request->password) ;
        }
        $user->save();

        /** assign role to the new admin */
        DB::table('model_has_roles')->where('model_id',$user->id)->delete();
        $user->assignRole($request->role_name);

        $admin=new UserResource($user);

        return response()->json(['admin'=>$admin,'message'=>trans('api.admin data updated successfully')]);
    }

    public function getUserAccount($userId){
        $user = User::findOrFail($userId);
        $user = new UserResource ($user);
        return response()->json(['userData'=>$user,'message'=>trans('api.user data')]);
    }

}
