<?php

namespace App\Http\Controllers\Api\User;

use App\Actions\UserExperimentalSubscription;
use App\Helpers\Helper;
use App\Helpers\HelperNotification;
use App\Http\Controllers\Controller;
use App\Models\AdminDashboard\School\SchoolEmployee;
use FontLib\Table\Type\name;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

use Tymon\JWTAuth\Exceptions\JWTException;
use Str;
use Hash;
use App\Models\User;
use App\Models\OtpVerification;
use App\Traits\SmsTrait;
use Illuminate\Support\Facades\Log;
use App\Http\Resources\User\UserResource;
use JWTAuth;
use Carbon\Carbon;
use Mail;
use Illuminate\Support\Facades\URL;

class AuthController extends Controller
{
    use SmsTrait;

    /**
     * function that validate token before login
     */
    public function validateToken(Request $request)
    {
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json(['message' => trans('api.Token not provided')], 401);
        }

        try {
            // Validate the token
            JWTAuth::setToken($token)->checkOrFail();
            $user = JWTAuth::setToken($token)->authenticate();
        } catch (JWTException $e) {
            return response()->json(['message' => trans('api.Invalid token')], 401);
        }

        /** return current semester */
        $currentSemester = Helper::getCurrentSchoolSemester();

        return response()->json(['user' => new UserResource($user), 'current_semester'=>$currentSemester,'message' => trans('api.user data')]);
    }

    /*
        Login function
        user can login with email or phone number
    */
    public function authenticate(Request $request)
    {
        $credentials=[];
        if (is_numeric($request->get('email'))) {
            $credentials = ['phone' => $request->get('email'), 'password' => $request->get('password')];
        } elseif (filter_var($request->get('email'), FILTER_VALIDATE_EMAIL)) {
            $credentials = ['email' => $request->get('email'), 'password' => $request->get('password')];
        }

        try {
            if ($credentials) {
                if (! $token = JWTAuth::attempt($credentials)) {
                    return response()->json(['message' => trans('api.invalid_credentials')], 401);
                }
            } else {
                return response()->json(['message' => trans('api.invalid_credentials')], 404);
            }
        } catch (JWTException $e) {
            return response()->json(['message' => trans('api.could_not_create_token')], 500);
        }

        $user = JWTAuth::user($token);

        $userTypes = explode(",", $user->user_type);

        if ($request->board_type == 'super_admin') {
            if ($request->board_type == 'super_admin' && !in_array('super_admin', $userTypes)) {
                return response()->json(['message' => trans('api.invalid_credentials')], 401);
            }
        }

        if ($request->board_type == 'school_manager') {
            if ($request->board_type == 'school_manager' && (!in_array('school_manager', $userTypes) && !in_array('teacher', $userTypes))) {
                return response()->json(['message' => trans('api.invalid_credentials')], 401);
            }
        }

        $user->verification_code = mt_rand(1000, 9999);
        $user->save();

        if ($user->status == 'active') {

            /** send otp */
            $this->createOtp($user->phone , $user->verification_code);
            $this->sendOTP($user->phone, $user->verification_code);

            $user = new UserResource($user);
            $message_text = "للدخول الى النظام: رمز التحقق الخاص بك للدخول الى النظام هو ".'[ '. $user->verification_code.' ]';
            $this->sendSMSNotification($user->phone, $message_text, $user->manager_id);
            return response()->json(['otp' => $user->verification_code, 'phone' => $user->phone, 'user_types' => $userTypes, 'message' => trans('api.verfiy login via otp login verfication endpoint')], 200);
        } else {
            /** send otp */
            $this->createOtp($user->phone , $user->verification_code);
            $this->sendOTP($user->phone, $user->verification_code);
            $message_text = "شكرا لتسجيلك! رمز التحقق الخاص بك لانشاء الحساب هو " . '[ ' . $user->verification_code . ' ]';
            $this->sendSMSNotification($user->phone, $message_text, $user->manager_id);
            return response()->json(['otp' => $user->verification_code, 'phone' => $user->phone, 'user_types' => $userTypes, 'verification_code' => $user->verification_code, 'message' => trans('api.account no verified yet')], 403);
        }
    }

    /*
        function to verfiy login account user
    */
    public function verfiyLogin(Request $request)
    {

        // تم التعديل ليقبل اي كود تفعيل مؤقتا\
        
        // $otpValidationResult = $this->verifyOtp($request->phone, $request->otp);

        // // If OTP validation fails, return the response
        // if ($otpValidationResult !== true) {
        //     return $otpValidationResult;
        // }

        // $user =  User::where('phone', $request->phone)->where('verification_code', $request->otp)->first();
        
        $user =  User::where('phone', $request->phone)->first();


        if ($user) {
            $user->verification_code = mt_rand(1000, 9999);
            $user->save();

            $token = JWTAuth::fromUser($user);

            $user = new UserResource($user);

            /** return current semester */
            $currentSemester = Helper::getCurrentSchoolSemester();

            return response()->json(['userData' => $user, 'token' => $token, 'currentSemester' => $currentSemester, 'message' => trans('api.login verfied successfully')]);
        } else {
            return response()->json(['message' => trans('api.otp not valid')], 400);
        }
    }

    /*
        refresh token function to verfiy account user
    */
    public function refreshToken(Request $request)
    {
        try {
            $user = User::where('email', $request->email)->orwhere('phone', $request->email)->firstOrFail();
            if ($user) {
                $token = JWTAuth::fromUser($user);
                return response()->json(compact('token'));
            } else {
                return response()->json(['error' => 'invalid_credentials'], 401);
            }
        } catch (JWTException $e) {
            return response()->json(['error' => 'could_not_create_token'], 500);
        }
    }

    /* registeration function */
    public function register(Request $request)
    {
        $user = User::where('status', 'inactive')->where('email', $request->email)->orWhere('phone', $request->phone)->first();

        if ($user && $user->status == 'inactive') {

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
                'phone' => 'required|string|regex:/^05\d{8}$/|unique:users,phone,' . $user->id,
                'password' => 'required|string|min:8|confirmed',
            ]);

            if ($validator->fails()) {
                $errors = $validator->errors();
                foreach ($errors->all() as $message) {
                    $error[] =  trans("api.$message");
                }
                return response()->json(['message' => trans('api.error entering data'), 'errors' => $validator->errors()], 400);
            }

            $user->name = $request->name;
            $user->email = $request->email;
            $user->phone = $request->phone;
            $user->password = Hash::make($request->password);
            $user->status = 'inactive';
            $user->user_type = 'school_manager';
            $user->verification_code = mt_rand(1000, 9999);
            $user->save();

            /**  add school manager as a first school adminstrative */
            $adminstrative = SchoolEmployee::where('user_id', $user->id)->first();
            $adminstrative->type = 'adminstrative';
            $adminstrative->name = $user->name;
            $adminstrative->email = $user->email;
            $adminstrative->phone = $user->phone;
            $adminstrative->current_work_id = 1;
            $adminstrative->have_account = 'yes';
            $adminstrative->user_id = $user->id;
            $adminstrative->save();

            $token = JWTAuth::fromUser($user);

            /** send otp */
            $this->createOtp($user->phone , $user->verification_code);
            $this->sendOTP($user->phone, $user->verification_code);

            $user = new UserResource($user);
            /* save user experimental subscription plans*/
            UserExperimentalSubscription::excute($user);

            return response()->json(['otp' => $user->verification_code, 'userData' => $user, 'message' => trans('api.new user registered verfication code will be sent in a few seconds')]);
        } else {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users',
                'phone' => 'required|string|regex:/^05\d{8}$/|unique:users',
                'password' => 'required|string|min:8|confirmed',
            ]);

            if ($validator->fails()) {
                $errors = $validator->errors();
                foreach ($errors->all() as $message) {
                    $error[] =  trans("api.$message");
                }
                return response()->json(['message' => trans('api.error entering data'), 'errors' => $validator->errors()], 400);
            }

            $user = new User();
            $user->name = $request->name;
            $user->email = $request->email;
            $user->phone = $request->phone;
            $user->password = Hash::make($request->password);
            $user->status = 'inactive';
            $user->user_type = 'school_manager';
            $user->verification_code = mt_rand(1000, 9999);
            $user->save();

            /**  add school manager as a first school adminstrative */
            $adminstrative = new SchoolEmployee();
            $adminstrative->type = 'adminstrative';
            $adminstrative->name = $user->name;
            $adminstrative->email = $user->email;
            $adminstrative->phone = $user->phone;
            $adminstrative->current_work_id = 1;
            $adminstrative->have_account = 'yes';
            $adminstrative->user_id = $user->id;
            $adminstrative->save();

            $token = JWTAuth::fromUser($user);

            /** send otp */
            $this->createOtp($user->phone , $user->verification_code);
            $this->sendOTP($user->phone, $user->verification_code);

            // register role school-manger//
            $user->assignRole('school-manager');


            $user = new UserResource($user);
            /* save user experimental subscription plans*/
            UserExperimentalSubscription::excute($user);

            /** send verfication email */

            $verificationUrl = URL::temporarySignedRoute(
                'verification.verify',
                now()->addMinutes(30),
                ['id' => $user->id, 'hash' => sha1($user->email)]
            );

            // Send email (use a mailer or notification)
            $data = ['verificationUrl' => $verificationUrl];

            Mail::send('emails/email-verification', $data, function($msg) use ($user) {
                $msg->to($user->email, 'Laam App')->subject('Email Verification');
                $msg->from(config('mail.from.address'), config('mail.from.name'));
            });

            return response()->json(['otp' => $user->verification_code, 'userData' => $user, 'message' => trans('api.new user registered verfication code will be sent in a few seconds')]);
        }
    }


    public function accountVerfication(Request $request)
    {
        // تم التعديل ليقبل اي كود تفعيل مؤقتا\

        // $otpValidationResult = $this->verifyOtp($request->phone, $request->code);

        // // If OTP validation fails, return the response
        // if ($otpValidationResult !== true) {
        //     return $otpValidationResult;
        // }

        // $user =  User::where('phone',$request->phone)->where('verification_code', $request->code)->first();
        
        $user =  User::where('phone',$request->phone)->first();

        
        if ($user) {
            $user->status = 'active';
            $user->verification_code = mt_rand(1000, 9999);
            $user->save();

            $token = JWTAuth::fromUser($user);
            $user = new UserResource($user);
            return response()->json(['userData' => $user, 'token' => $token, 'message' => trans('api.account verfied successfully')]);
        } else {
            return response()->json(['errors' => trans('api.phone and verfication code not valid')], 400);
        }
    }

    public function getUserProfile(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }

        $token = JWTAuth::fromUser($user);
        $user = new UserResource($user);
        return response()->json(['userData' => $user, 'token' => $token, 'message' => trans('api.user data')]);
    }

    /////////// function update user profile ///////////////
    public function updateUserProfile(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }

        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone = $request->phone;
        $user->remember_token = $request->remember_token;
        $user->save();

        $token = JWTAuth::fromUser($user);
        $user = new UserResource($user);

        return response()->json(['userData' => $user, 'token' => $token, 'message' => trans('api.user data updated successfully')]);
    }


    /////////////// public function send reset email ////////
    public function sendResetPasswordOtpViaEmail(Request $request){
        try {
            $user = User::where('email', $request->email)->first();
            if(!$user){
                return response()->json(['message' => trans('api.email not found')], 400);
            }

            if($user->email_verified_at == NULL){
                return response()->json(['message' => trans('api.email will not be sent cause of email not verified')], 400);
            }
            $user->verification_code = mt_rand(1000, 9999);
            $user->save();

            $this->createOtp($user->phone , $user->verification_code);
            ////// send reset password email////////
            $data = ['otp' => $user->verification_code];

            Mail::send('emails/reset_password', $data, function($msg) use ($user) {
                $msg->to($user->email, 'Laam App')->subject('Password Reset');
                $msg->from(config('mail.from.address'), config('mail.from.name'));
            });

            return response()->json(['otp' => $user->verification_code, 'message' => trans('api.reset password otp sent to email successfully')]);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('api.email not found')], 400);
        }
    }

    /////////////// public function send reset sms ////////
    public function sendResetPasswordOtpViaSms(Request $request)
    {
        try {
            $user = User::where('phone', $request->phone)->firstOrFail();
            $user->verification_code = mt_rand(1000, 9999);
            $user->save();
            /** send otp */
            $this->createOtp($user->phone , $user->verification_code);
            $this->sendOTP($user->phone, $user->verification_code);


            return response()->json(['otp' => $user->verification_code, 'message' => trans('api.reset password otp sent via sms successfully')]);
        } catch (\Exception $e) {
            return response()->json(['message' => trans('api.no user found with this phone number')], 422);
        }
    }

    /** reset password  function */
    public function resetPasswordVerfication(Request $request){

        // تم التعديل ليقبل اي كود تفعيل مؤقتا\

        // $user = User::where('verification_code', $request->otp)->where('phone',$request->phone)->orWhere('email', $request->phone)->firstOrFail();

        // $otpValidationResult = $this->verifyOtp($user->phone, $request->otp);

        // // If OTP validation fails, return the response
        // if ($otpValidationResult !== true) {
        //     return $otpValidationResult;
        // }
        
        $user = User::where('phone',$request->phone)->orWhere('email', $request->phone)->firstOrFail();


        $user->verification_code = mt_rand(1000, 9999);
        $user->save();

        $token = JWTAuth::fromUser($user);
        $user = new UserResource($user);
        return response()->json(['userData' => $user, 'token' => $token, 'message' => trans('api.redirect to change password')]);
    }

    public function resetPassword(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }

        $validator = Validator::make($request->all(), [
            'password' => 'required|string|min:8',
        ]);

        $user->password = Hash::make($request->password);
        $user->password_reseted = 'yes';
        $user->save();
        return response()->json(['message' => trans('api.password rested successfully')]);
    }


    /** function that remove user data */
    public function removeUser(request $request)
    {
        $user = User::where('phone',$request->phone)->firstOrFail();

        if ($user) {
            $user->delete();
            return response()->json(['message' => trans('api.user deleted successfully')]);
        } else {
            return response()->json(['message' => trans('api.no user found with this number')], 404);
        }
    }

    /** function that remove user data */
    public function resndOtp(request $request)
    {
        $user =  User::where('phone', $request->phone)->firstOrFail();
        $user->verification_code = mt_rand(1000, 9999);
        $user->save();

        /** send otp */
        $this->createOtp($user->phone , $user->verification_code);
        $this->sendOTP($user->phone, $user->verification_code);

        return response()->json(['verification_code' => $user->verification_code, 'message' => trans('api.otp sent successfully')]);
    }


    public function createOtp($phone , $otp){
        /** remove any old otp records*/
        $otpRecord = OtpVerification::where('phone', $phone)->delete();

        /** create new otp*/
        OtpVerification::updateOrCreate(
            ['phone' => $phone],
            [
                'otp' => $otp,
                'expires_at' => Carbon::now()->addMinutes(5)
            ]
        );
    }

    public function verifyOtp($phone, $otp){
        $otpRecord = OtpVerification::where('phone', $phone)->first();

        if (!$otpRecord) {
            return response()->json(['message' => trans('api.Phone number not found or OTP expired.')], 400);
        }

        if ($otpRecord->otp != $otp) {
            return response()->json(['message' => trans('api.Invalid OTP.')], 400);
        }

        if (Carbon::now('Asia/Riyadh')->greaterThan($otpRecord->expires_at)) {
            return response()->json(['message' => trans('api.OTP has expired.')], 400);
        }

        $otpRecord->delete();
        return true;
    }


}
