<?php

namespace App\Http\Controllers\Api\TeacherDashboard;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\SemesterWeekDay;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentAttendance;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;
use Illuminate\Validation\Rules\Password;

class TeacherSettingController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:teacher-dashboard-settings']);
    }

    public function changePassword(Request $request)
    {
        // Validate incoming request (optional but recommended)
        $request->validate([
            'current_password' => 'required',
            'new_password' => ['required','confirmed', Password::min(8)->mixedCase()->letters()->numbers()->symbols()],
        ]);

        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }

        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'message' => trans('api.validation_error'),
                'errors' => [
                    'current_password' => [
                        trans('api.current password is incorrect'),
                    ],
                ],
            ], 404);
        }

        if ($request->current_password == $request->new_password) {
            return response()->json([
                'message' => trans('api.validation_error'),
                'errors' => [
                    'current_password' => [
                        trans('api.current and new password are the same'),
                    ],
                ],
            ], 404);
        }

        $user->update(['password' => Hash::make($request->new_password)]);

        return response()->json(['message' => trans('api.password changed successfully')]);
    }

    public function indexPage($schoolId){
        $school = School::findOrFail($schoolId);
        $semester = Helper::getCurrentSchoolSemester();

        $today = Carbon::today();
        $weekday = $today->format('D');
        if ($weekday === 'Fri') {
            $today->addDays(2);
        } elseif ($weekday === 'Sat') {
            $today->addDay();
        }
        $semesterWeekDay = SemesterWeekDay::where('semester_id', $semester->id)->where('date', $today->format('d/m/Y'))->first();
        $days_before_today_count = SemesterWeekDay::where('semester_id', $semester->id)->where('date', '<', date('d/m/Y'))->count();


        $schoolClassesCount = SchoolClass::where('school_id',$school->id)->count();
        $schoolStudentsCount=Student::where('school_id',$school->id)->count();

        $studentIds =Student::where('school_id',$school->id)->pluck('id')->toArray();
        $absent = StudentAttendance::whereIn('student_id',$studentIds)->where('date',date('d/m/Y'))->whereIn('status',['absent','absent_with_excuse'])->count();
        $attended = StudentAttendance::whereIn('student_id',$studentIds)->where('date',date('d/m/Y'))->where('status','attended')->count();

        $data =[
            'school_name'=>$school->name,
            'semester'=>$semester->name,
            'semester_total_days'=>$semester->period,
            'semester_remaining_days'=>$semester->period - $days_before_today_count,
            'week'=>$semesterWeekDay->week?->name ,
            'weekday'=>$weekday,
            'current_school_year'=> $school->current_school_year,
            'schoolClassesCount'=>$schoolClassesCount,
            'schoolStudentsCount'=>$schoolStudentsCount,
            'absent_count'=>$absent,
            'attended_count'=>$attended,
        ];

        return response()->json(['data'=>$data,'message' => trans('api.teacher index page informations')]);

    }
}
