<?php

namespace App\Http\Controllers\Api\School;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\School\SchoolStudentMoreInfoResource;
use App\Http\Resources\School\SchoolStudentResource;
use Illuminate\Http\Request;
use App\Http\Requests\School\StudentRequest;
use App\Models\AdminDashboard\School\SchoolSemester;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\AbsenseAndLateness\PeriodStudentAttendance;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentLateness;
use Carbon\Carbon;

class StudentController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api',,'permission:students']);
    }


    /** function store school student informations */
    public function store(StudentRequest $request)
    {
        $data = $request->all();
        $data['current_school_year'] = Helper::getSchoolYear();
        $student = Student::create($data);
        $student = new SchoolStudentResource($student);
        return response()->json(['student' => $student, 'message' => trans('api.record stored successfully')]);
    }

    /** function show school student informations */
    public function show(Request $request)
    {
        $student = Student::findOrFail($request->student_id);
        $student = new SchoolStudentMoreInfoResource($student);
        return response()->json(['student' => $student, 'message' => trans('api.record details')]);
    }

    public function getStudentAbsenceLog(Student $student)
    {
        $absenceData = PeriodStudentAttendance::where('student_id', $student->id)
            ->where('school_id', $student->school_id)
            ->where('class_id', $student->class_id)
            ->whereIn('status', ['absent', 'absent_with_excuse'])
            ->get();

        // Set locale to Arabic
        Carbon::setLocale('ar');

        // Define the first 3 semesters statically
        $staticSemesters = [
            1 => ['semester_name' => 'الفصل الدراسي الأول', 'absent_count' => 0, 'periods' => []],
            2 => ['semester_name' => 'الفصل الدراسي الثاني', 'absent_count' => 0, 'periods' => []],
            3 => ['semester_name' => 'الفصل الدراسي الثالث', 'absent_count' => 0, 'periods' => []],
        ];

        // Group data by `semester_id`
        $groupedData = $absenceData->groupBy('semester_id')->map(function ($group, $semesterID) {
            return [
                'semester_name' => SchoolSemester::find($semesterID)->name,
                'absent_count' => $group->count(),
                'periods' => $group->map(function ($item) {
                    return [
                        'period_name' => $item->period->name,
                        'date' => Carbon::createFromFormat('d/m/Y', $item->date)->translatedFormat('l d F Y'),
                    ];
                })->values(),
            ];
        });

        // Merge static semesters with data from groupedData, ensuring first 3 semesters are present
        $finalData = $staticSemesters;

        foreach ($groupedData as $semesterID => $data) {
            // If semester ID is within the first 3, replace static data with actual data
            if (isset($finalData[$semesterID])) {
                $finalData[$semesterID] = $data;
            } else {
                // If it's beyond the first 3 semesters, add it dynamically
                $finalData[$semesterID] = $data;
            }
        }

        // Ensure the final data is ordered by semester ID
        ksort($finalData);

        return response()->json(['absenceLogs' => array_values($finalData), 'message' => trans('api.student absence Logs')]);
    }



    /** function show school student lateness log */
    public function getStudentLatenessLog(Student $student)
    {
        $latenessData = StudentLateness::where('student_id', $student->id)
            ->where('school_id', $student->school_id)
            ->where('class_id', $student->class_id)
            ->whereIn('status', ['late', 'late_with_excuse'])
            ->get();

        // Set locale to Arabic
        Carbon::setLocale('ar');

        // Define the first 3 semesters statically
        $staticSemesters = [
            1 => ['semester_name' => 'الفصل الدراسي الأول', 'lateness_count' => 0, 'dates' => []],
            2 => ['semester_name' => 'الفصل الدراسي الثاني', 'lateness_count' => 0, 'dates' => []],
            3 => ['semester_name' => 'الفصل الدراسي الثالث', 'lateness_count' => 0, 'dates' => []],
        ];

        // Group data by `semester_id`
        $groupedData = $latenessData->groupBy('semester_id')->map(function ($group, $semesterID) {
            return [
                'semester_name' => SchoolSemester::find($semesterID)->name,
                'lateness_count' => $group->count(),
                'dates' => $group->map(function ($item) {
                    return [
                        'date' => Carbon::createFromFormat('d/m/Y', $item->date)->translatedFormat('l d F Y'),
                    ];
                })->values(),
            ];
        });

        // Merge static semesters with data from groupedData, ensuring first 3 semesters are present
        $finalData = $staticSemesters;

        foreach ($groupedData as $semesterID => $data) {
            // If semester ID is within the first 3, replace static data with actual data
            if (isset($finalData[$semesterID])) {
                $finalData[$semesterID] = $data;
            } else {
                // If it's beyond the first 3 semesters, add it dynamically
                $finalData[$semesterID] = $data;
            }
        }

        // Ensure the final data is ordered by semester ID
        ksort($finalData);

        return response()->json(['latenessLogs' => array_values($finalData), 'message' => trans('api.student lateness Logs')]);
    }


    /** function update school student informations */
    public function update(StudentRequest $request, $id)
    {
        $student = Student::findOrFail($id);
        $student->update($request->all());
        $student = new SchoolStudentResource($student);
        return response()->json(['student' => $student, 'message' => trans('api.record updated successfully')]);
    }

    /* this method will delete specific Model recored
        by using ID
    */
    public function destroy(Request $request)
    {
        Student::findOrFail($request->student_id)->forceDelete();
        return response()->json(['message' => trans('api.record deleted successflly')]);
    }
}
