<?php

namespace App\Http\Controllers\Api\School;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Models\ManagerDashboard\Tables\ShiftSchedule\ShiftSchedule;
use Illuminate\Http\Request;
use App\Http\Requests\School\SchoolSemesterRequest;
use App\Http\Resources\School\SchoolSemesterResource;
use App\Models\AdminDashboard\School\SchoolSemester;
use App\Models\AdminDashboard\School\SemesterWeekDay;
use Carbon\Carbon;

class SchoolSemesterController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct(){
        //$this->middleware(['auth:api','permission:school-semesters']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request){
        // Start with the base query
        $semester = Helper::getCurrentSchoolSemester();
        
        $startDate = Carbon::createFromFormat('d/m/Y', $semester->start_date);
       

        $data = SchoolSemester::query();

          // Filter semesters after the current semester
        $data = $data->whereRaw("STR_TO_DATE(start_date, '%d/%m/%Y') >= ?", [$startDate->format('Y-m-d')]);

          
        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            return !in_array($key,['start_date','end_date','start_date_type','end_date_type','per_page']);
        }, ARRAY_FILTER_USE_KEY);

        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\SchoolSemester', $filterData);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        if (isset($queryParams['start_date_type']) && isset($queryParams['start_date'])) {
            $start_date = $queryParams['start_date'];
            if ($queryParams['start_date_type'] == "hijri") {
                $start_date = Helper::getCoptic($queryParams['start_date']);
            }
            $data = $data->where('start_date', 'like', "%$start_date%");
        }

        if (isset($queryParams['end_date_type']) && isset($queryParams['end_date'])) {
            $end_date = $queryParams['end_date'];
            if ($queryParams['end_date_type'] == "hijri") {
                $end_date = Helper::getCoptic($queryParams['end_date']);
            }
            $data = $data->where('end_date', 'like', "%$end_date%");
        }

        $filterData = SchoolSemesterResource::collection($data->where('status', '1')->paginate($perPage));

        return response()->json([
            'schoolSemesters' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored
        by using ID
    */
    public function getById(SchoolSemester $schoolSemester)
    {
        $schoolSemester = new SchoolSemesterResource($schoolSemester);
        return response()->json(['schoolSemester'=>$schoolSemester,'message'=>trans('api.semester details')],200);
    }

    /* this method will stord data to DB Model*/
    public function store(SchoolSemesterRequest $request)
    {
        $start_date = \Carbon\Carbon::createFromFormat('Y-m-d',$request->start_date)->format('d/m/Y');
        $end_date = \Carbon\Carbon::createFromFormat('Y-m-d',$request->end_date)->format('d/m/Y');

        $schoolSemester = new SchoolSemester();
        $schoolSemester->name = $request->name;
        $schoolSemester->start_date = $start_date;
        $schoolSemester->end_date = $end_date;
        $schoolSemester->start_date_type = 'coptic';
        $schoolSemester->end_date_type = 'coptic';

        //get the period in days between start_date and end_date
        $start_date = \Carbon\Carbon::parse($request->start_date);
        $end_date = \Carbon\Carbon::parse($request->end_date);

        $days = $start_date->diffInDays($end_date);

        $schoolSemester->period = $days+1;
        $schoolSemester->status = $request->status;
        $schoolSemester->save();

        $totalDays = $start_date->diffInDays($end_date);
        $totalWeeks = intval($totalDays / 7);

        if ($totalWeeks  == 13) {
            /**save  semester week days*/
            Helper::saveSemesterWeekDays($schoolSemester->id,$schoolSemester->start_date,$schoolSemester->end_date);

            /**  save shift_schedules */
            Helper::saveShiftSchedules($schoolSemester->id,$schoolSemester->start_date,$schoolSemester->end_date);

            return response()->json(['message'=>trans('api.record stored successflly')],200);

        }else{
            SchoolSemester::find($schoolSemester->id)->delete();

            return response()->json([
                'message' => trans('api.validation_error'),
                'total_weeks'=>$totalWeeks,
                'errors' => [
                    'semester_weeks' => [
                        trans('api.The provided semester period does not cover 13 weeks')
                    ]
                ]
            ], 400);

        }
    }

    /* this method will update specific Model recored
        by using ID
    */
    public function update(Request $request,$schoolSemesterId)
    {
        if (isset($request['start_date_type']) && isset($request['end_date_type']) && isset($request['start_date'])  && isset($request['end_date'])) {
            $start_date = $request['start_date'];
            $end_date = $request['end_date'];

            if ($request['date_type'] == "hijri") {
                $start_date = Helper::getCoptic($request['start_date']);
                $end_date = Helper::getCoptic($request['end_date']);
            }
        }

        $start = Carbon::createFromFormat('d/m/Y', $start_date);
        $end = Carbon::createFromFormat('d/m/Y', $end_date);
        $totalDays = $start->diffInDays($end);
        $totalWeeks = intval($totalDays / 7);

        if ($totalWeeks  == 13) {
            $schoolSemester = SchoolSemester::findOrFail($schoolSemesterId);
            $schoolSemester->start_date = $request->start_date;
            $schoolSemester->end_date = $request->end_date;
            $schoolSemester->start_date_type = $request->start_date_type;
            $schoolSemester->end_date_type = $request->end_date_type;
            $schoolSemester->period = $totalDays+1;
            $schoolSemester->save();

            /**update  new semester week days*/
            Helper::saveSemesterWeekDays($schoolSemester->id,$schoolSemester->start_date,$schoolSemester->end_date);

            /**  save shift_schedules */
            Helper::saveShiftSchedules($schoolSemester->id,$schoolSemester->start_date,$schoolSemester->end_date);

            return response()->json(['message'=>trans('api.semester period updated successflly')],200);

        }else{
            return response()->json([
                'message' => trans('api.validation_error'),
                'total_weeks'=>$totalWeeks,
                'errors' => [
                    'semester_weeks' => [
                        trans('api.The provided semester period does not cover 13 weeks')
                    ]
                ]
            ], 400);
        }
    }

    /* this method will change activation status for
       specific Model recored by using ID
    */
    public function changeStatus(Request $request,SchoolSemester $schoolSemester)
    {
        $schoolSemester->status = $request->status;
        $schoolSemester->save();
        return response()->json(['message'=>trans('api.status changed successflly')],200);
    }

    /* this method will archive specific Model recored
        by using ID
    */
    public function archive(SchoolSemester $schoolSemester)
    {
        $schoolSemester->delete();
        return response()->json(['message'=>trans('api.record archived successflly')],200);
    }

    /* this method will delete specific Model recored
        by using ID
    */
    public function destroy(SchoolSemester $schoolSemester)
    {
        $schoolSemester->forceDelete();
        return response()->json(['message'=>trans('api.record deleted successflly')],200);
    }

    /* this method will restore specific Model recored
        by using ID
    */
    public function restore(SchoolSemester $schoolSemester)
    {
        $schoolSemester->restore();
        return response()->json(['message'=>trans('api.record restored successflly')],200);
    }
}
