<?php

namespace App\Http\Controllers\Api\School;

use App\Helpers\Helper;
use App\Helpers\HelperNotification;
use App\Http\Controllers\Controller;
use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\SchoolEmployeeAttendance;
use Illuminate\Http\Request;
use JWTAuth;
use App\Http\Resources\School\SchoolEmployeeAttendanceResource;
use Carbon\Carbon;

class SchoolEmployeeAttendanceController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:school-employee-attendance']);
    }

    public function getDayAttendance(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }
        $school = School::where('user_id',$user->id)->where('id',$request->school_id)->firstOrFail();
        $schoolEmployeeIds = SchoolEmployee::where('school_id',$school->id)->where('type',$request->type)->pluck('id')->toArray();

        if (isset($request['date'])) {
            $date = $request['date'];
            if ($request['date_type'] == "hijri") {
                $date = Helper::getCoptic($request['date']);
            }
        }

        // Paginate results with a default value if not provided
        $perPage = $request->per_page ?? 15;
        $schoolEmployeesAttendance = SchoolEmployeeAttendance::whereIn('school_employee_id',$schoolEmployeeIds)->where('date',$date)->paginate($perPage);

        if($request->status){
            $schoolEmployeesAttendance = SchoolEmployeeAttendance::whereIn('school_employee_id',$schoolEmployeeIds)->where('date',$date)->where('status',$request->status)->paginate($perPage);
        }
        $schoolEmployeesAttendance = SchoolEmployeeAttendanceResource::collection($schoolEmployeesAttendance);

        $attendedSchoolEmployees = SchoolEmployeeAttendance::whereIn('school_employee_id',$schoolEmployeeIds)->where('status','attended')->where('date',$date)->count();
        $absentSchoolEmployees = SchoolEmployeeAttendance::whereIn('school_employee_id',$schoolEmployeeIds)->where('status','absent')->where('date',$request->date)->count();
        $lateSchoolEmployees = SchoolEmployeeAttendance::whereIn('school_employee_id',$schoolEmployeeIds)->where('status','late')->where('date',$request->date)->count();

        return response()->json([
            'attendedSchoolEmployees'=>$attendedSchoolEmployees,
            'absentSchoolEmployees'=>$absentSchoolEmployees,
            'lateSchoolEmployees'=>$lateSchoolEmployees,
            'schoolEmployeesAttendance'=>$schoolEmployeesAttendance,
            'message'=>trans('api.list of employee attendance'
            )]);

    }

    /** store school attendance */
    public function store(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }

        /** make date always coptic when insert */
        if (isset($request['date'])) {
            $date = $request['date'];
            if ($request['date_type'] == "hijri") {
                $date = Helper::getCoptic($request['date']);
            }
        }else{
            $date = date('d/m/Y');
        }

        $attendanceDate = Carbon::createFromFormat('d/m/Y', $date);
        $currentDate = Carbon::now('Asia/Riyadh');

        // Check if the day is Friday or Saturday
        $currentDay = $attendanceDate->format('l'); // Get the day name
        if (in_array($currentDay, ['Friday', 'Saturday'])) {
            return response()->json(['message' => trans('api.Not able to make attendance on Friday or Saturday.')], 400);
        }

        // Check if the date is in the future
        if ($attendanceDate->greaterThan($currentDate)) {
            return response()->json(['message' => trans('api.Attendance date cannot be in the future')], 400);
        }

        $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $schoolEmployees = SchoolEmployee::where('school_id', $school->id)->where('type', $request->type)->get();

        $checkIfCreated = SchoolEmployeeAttendance::where('type', $request->type)
            ->where('date', $attendanceDate->format('d/m/Y'))
            ->where('school_id', $school->id)
            ->exists();

        if (!$checkIfCreated) {
            foreach ($schoolEmployees as $schoolEmployee) {
                $schoolEmployeeAttendance = new SchoolEmployeeAttendance();
                $schoolEmployeeAttendance->school_id = $school->id;
                $schoolEmployeeAttendance->school_employee_id = $schoolEmployee->id;
                $schoolEmployeeAttendance->status = 'attended';
                $schoolEmployeeAttendance->date = $attendanceDate->format('d/m/Y'); // Format date here
                $schoolEmployeeAttendance->date_type = 'coptic';
                $schoolEmployeeAttendance->type = $request->type;
                $schoolEmployeeAttendance->save();
            }
            return response()->json(['message' => trans('api.attendance added successfully')]);
        }
        return response()->json(['message' => trans('api.attendance added before')]);
    }



    /** change status of attendance to particular employee */
    public function changeAttendanceStatus(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }
        $school = School::where('user_id',$user->id)->where('id',$request->school_id)->firstOrFail();
        $attendanceData = $request->input('attendance_data'); // This would be your array of JSON objects

        if (isset($request['date'])) {
            $date = $request['date'];
            if ($request['date_type'] == "hijri") {
                $date = Helper::getCoptic($request['date']);
            }
        }else{
            return response()->json(['message'=>trans('api.date field is required')],400);
        }

        if ($attendanceData && is_array($attendanceData)) {
            foreach ($attendanceData as $attendance) {
                $schoolEmployeeAttendance = SchoolEmployeeAttendance::where('school_employee_id',$attendance['school_employee_id'])->where('date',$request->date)->first();
                if ($attendance['status'] == "attended") {
                    // حذف اتمني حصص االنتظار عند حضور املعلم بعد التأخر
                    HelperNotification::create_waiting_periods_notifications($request->school_id,$attendance['school_employee_id'], "attended");
                    $schoolEmployee_check = SchoolEmployeeAttendance::where('school_employee_id', $attendance['school_employee_id'])->latest()->first();
                    if($schoolEmployee_check->status=="absent"){
                        // مساءلة عند غياب املعلم )عندما حيضر املعلم بعد الغياب
                        HelperNotification::create_employee_late_notifications($request->school_id,$attendance['school_employee_id'], "absent");
                    }
                }elseif($attendance['status'] == "late"){
                    HelperNotification::create_employee_late_notifications($request->school_id,$attendance['school_employee_id'], "late");
                }elseif($attendance['status'] == "absent"){
                    HelperNotification::create_waiting_periods_notifications($request->school_id,$attendance['school_employee_id'], "absent");
                }
                $schoolEmployeeAttendance->update([
                    'status' => $attendance['status'],
                    'absence_reason' => $attendance['absence_reason'],
                    'other_absence_reason' => $attendance['other_absence_reason'],
                    'late_reason' => $attendance['late_reason'],
                    'late_from_time' => $attendance['late_from_time'],
                    'late_to_time' => $attendance['late_to_time'],
                ]);
            }
            return response()->json(['message' => 'employee attendance updated successfully']);
        }
        return response()->json(['error' => 'Invalid data provided.'], 400);
    }

    /** autocomplete the school employee name */
    public function schoolEmployeeAutocompleteName(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }
        $school = School::where('user_id',$user->id)->where('id',$request->school_id)->firstOrFail();
        $results = SchoolEmployee::where('school_id',$school->id)->where('type',$request->type)->where('name', 'LIKE', "%$request->school_employee_name%")
            ->select('id','name')
            ->get();
        return response()->json($results);
    }

    /** get school employee attendance */
    public function getSchoolEmployeeAttendance(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }
        $school = School::where('user_id',$user->id)->where('id',$request->school_id)->firstOrFail();

        $schoolEmployeesAttendance = SchoolEmployeeAttendance::where('school_employee_id',$request->school_employee_id)->where('date',$request->date)->first();
        $schoolEmployeesAttendance = new SchoolEmployeeAttendanceResource($schoolEmployeesAttendance);
        return response()->json(['schoolEmployeesAttendance'=>$schoolEmployeesAttendance,'message'=>trans('api.school employee attendance data')]);
    }

    public function storeForMobApp(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }

        $school = School::where('user_id',$user->id)->where('id',$request->school_id)->firstOrFail();
        $schoolEmployeeIds = $request->school_employee_ids;
        $schoolEmployeesAttendance = $request->school_employee_attendance;

        $checkIfCreated = SchoolEmployeeAttendance::where('date',$request->date)->where('school_id', $school->id)->exists();
        if(!$checkIfCreated){
            foreach ($schoolEmployeeIds as $key=>$schoolEmployeeId){
                $schoolEmployeeAttendance = new SchoolEmployeeAttendance();
                $schoolEmployeeAttendance->school_id=$school->id;
                $schoolEmployeeAttendance->school_employee_id=$schoolEmployeeId;
                $schoolEmployeeAttendance->status=$schoolEmployeesAttendance[$key];
                $schoolEmployeeAttendance->date= $request->date;
                $schoolEmployeeAttendance->date_type=$request->date_type;
                $schoolEmployeeAttendance->save();
            }
            return response()->json(['message'=>trans('api.attendance added successfully')]);
        }
        return response()->json(['message'=>trans('api.attendance added before')]);
    }
}
