<?php

namespace App\Http\Controllers\Api\School;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\School\SchoolClassResource;
use App\Http\Resources\School\ClassNameResource;
use App\Http\Resources\School\CurrentWorkResource;
use App\Http\Resources\School\SchoolFacilitieResource;
use App\Http\Resources\School\SchoolReferenceResource;
use App\Http\Resources\School\SchoolResource;
use App\Http\Resources\School\SpecializationResource;
use App\Http\Resources\School\SchoolStudentResource;
use App\Http\Resources\School\SchoolEmployeeResource;
use App\Http\Resources\School\WeekResource;
use App\Models\User;
use App\Models\Week;
use Illuminate\Support\Facades\Hash;
use JWTAuth;

use Illuminate\Http\Request;

use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\SchoolReference;
use App\Models\AdminDashboard\School\SchoolFacilitie;
use App\Models\AdminDashboard\School\Student;

use App\Imports\StudentsImport;
use App\Imports\TeachersImport;
use App\Models\AdminDashboard\School\ClassName;
use App\Models\AdminDashboard\School\CurrentWork;
use App\Models\AdminDashboard\School\Specialization;
use App\Models\AdminDashboard\School\Subject;
use App\Models\ManagerDashboard\TeacherSchedule;
use App\Models\ManagerDashboard\TeacherScheduleData;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use Spatie\Permission\Models\Permission;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;


class SchoolController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api',,'permission:schools']);
    }


    /** function return school Students  */
    public function getSchools(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        // Start with the base query
        $data = School::query();
        // Retrieve all query parameters
        $queryParams = $request->query();
        // Apply filters if any
        if (!empty($queryParams)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\School', $queryParams);
        }
        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $filterData = SchoolResource::collection($data->paginate($perPage));

        return response()->json(['schools' => $filterData, 'message' => trans('api.all school records')]);
    }

    /** function return school informations */
    public function getSchoolInfo(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $school = School::findOrFail($request->school_id);
        $school = new SchoolResource($school);
        return response()->json(['school' => $school, 'message' => trans('api.school informations')]);
    }

    /** function return school Facilities */
    public function getSchoolFacilitiesInfo(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $school = School::findOrFail($request->school_id);
        $schoolFacilities = SchoolFacilitie::where('school_id', $school->id)->firstOrFail();
        $schoolFacilities = new SchoolFacilitieResource($schoolFacilities);
        return response()->json(['schoolFacilities' => $schoolFacilities, 'message' => trans('api.school facilities informations')]);
    }

    /** function return school References */
    public function getSchoolReferencesInfo(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $school = School::findOrFail($request->school_id);
        $schoolReferences = SchoolReference::where('school_id', $school->id)->firstOrFail();
        $schoolReferences = new SchoolReferenceResource($schoolReferences);
        return response()->json(['schoolReferences' => $schoolReferences, 'message' => trans('api.school references informations')]);
    }

    /** function return school Students  */
    public function getSchoolStudentsInfo(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $school = School::findOrFail($request->school_id);

        // Start with the base query
        $data = Student::query();
        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            return !in_array($key, ['per_page', 'name','class_name','class_id', 'class_number','class_numbers']);
        }, ARRAY_FILTER_USE_KEY);
        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\Student', $filterData);
        }
        $data->where('school_id', $school->id);

        // custom filter////
        if ($request->name) {
            $data->where('name',  'like','%'. $request->name . '%');
        }

        if ($request->class_name) {
            $classNameIds= ClassName::where('name',  'like', $request->class_name . '%')->pluck('id')->toArray();
            $schoolClasseIds = SchoolClass::where('school_id', $school->id)->whereIn('class_name_id', $classNameIds)->pluck('id')->toArray();
            $data->whereIn('class_id', $schoolClasseIds);
        }

        if ($request->class_id) {
            $className= ClassName::find($request->class_id);
            $schoolClasseIds = SchoolClass::where('school_id', $school->id)->where('class_name_id', $className->id)->pluck('id')->toArray();
            $data->whereIn('class_id', $schoolClasseIds);
        }

        if ($request->class_number) {
            $schoolClasseIds = SchoolClass::where('school_id', $school->id)->where('class_number', $request->class_number)->pluck('id')->toArray();
            $data->whereIn('class_id', $schoolClasseIds);
        }

        if ($request->class_numbers) {
            $schoolClasseIds = SchoolClass::where('school_id', $school->id)->whereIn('class_number',explode(',', $request->class_numbers) )->pluck('id')->toArray();
            $data->whereIn('class_id', $schoolClasseIds);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $filterData = SchoolStudentResource::collection($data->paginate($perPage));

        return response()->json(['schoolStudents' => $filterData, 'message' => trans('api.school Students records')]);
    }

    /** function return school Teachers  */
    public function getSchoolTeachrsInfo(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $school = School::findOrFail($request->school_id);

        // Start with the base query
        $data = SchoolEmployee::query();
        // Retrieve all query parameters
        $queryParams = $request->query();
        // Apply filters if any
        if (!empty($queryParams)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\SchoolEmployee', $queryParams);
        }

        $data =$data->where('school_id', $school->id)->where('type', 'teacher');
        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $filterData = SchoolEmployeeResource::collection($data->paginate($perPage));

        return response()->json(['schoolTeachers' => $filterData, 'message' => trans('api.school teachers records')]);
    }

    /** function return school Adminstratives  */
    public function getSchoolAdminstrativesInfo(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $school = School::findOrFail($request->school_id);

        // Start with the base query
        $data = SchoolEmployee::query();
        // Retrieve all query parameters
        $queryParams = $request->query();
        // Apply filters if any
        if (!empty($queryParams)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\SchoolEmployee', $queryParams);
        }

        $data->where('type', 'adminstrative')->where('school_id', $school->id);

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $filterData = SchoolEmployeeResource::collection($data->paginate($perPage));

        return response()->json(['schoolAdminstratives' => $filterData, 'message' => trans('api.school adminstratives records')]);
    }


    /** function update school informations */
    public function updateSchoolInfo(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        //$school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);
        $school->update($request->except('school_id'));
        return response()->json(['message' => trans('api.data updated successfully')]);
    }

    /** function update school Facilities */
    public function updateSchoolFacilitiesInfo(Request $request){
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();

        $school = School::findOrFail($request->school_id);

        $schoolFacility = SchoolFacilitie::where('school_id', $school->id)->firstOrFail();
        $schoolFacility->update($request->all());

        $school2 = School::where('user_id', $user->id)->where('id', '!=', $school->id)->first();

        if ($school2) {
            $school2Facility = SchoolFacilitie::where('school_id', $school2->id)->firstOrFail();
            $school2Facility->update($request->except('school_id'));
        }

        return response()->json(['message' => trans('api.data updated successfully')]);
    }

    /** function update school References */
    public function updateSchoolReferencesInfo(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);
        $schoolReference = SchoolReference::where('school_id', $school->id)->firstOrFail();
        $schoolReference->update($request->all());
        return response()->json(['message' => trans('api.data updated successfully')]);
    }

    /** function update school Students  */
    public function updateSchoolStudentsInfo(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();

        $school = School::findOrFail($request->school_id);

        /** get ids of old data */
        $classNameIds = ClassName::where('school_id',$school->id)->pluck('id')->toArray();
        $schoolClassIds = SchoolClass::where('school_id',$school->id)->pluck('id')->toArray();
        $studentIds = Student::where('school_id',$school->id)->pluck('id')->toArray();

        $file = $request->file('file');
        $skipRows = [0, 1, 2, 6, 5, 9, 10, 13, 14, 15, 16, 17, 18]; // Define the rows to skip
        Excel::import(new StudentsImport($file, $skipRows, $request->school_id), $file);

        $students = Student::where('school_id', $school->id)->pluck('id')->toArray();
        if (count($students) <= count($studentIds)) {
            return response()->json(['message' => trans('api.students data not imported check file is a valid file')], 404);
        }

        /* mark classes and class name data_imported flag to 1*/
        $classNames=ClassName::where('school_id',$school->id)->where('data_imported',0)->get();
        foreach($classNames as $className){
            $className ->update(['data_imported' => 1]);
        }

        $SchoolClasses=SchoolClass::where('school_id',$school->id)->where('data_imported',0)->get();
        foreach($SchoolClasses as $SchoolClass){
            $SchoolClass ->update(['data_imported' => 1]);
        }

        /** delete school old records before upload new file */
        ClassName::whereIn('id',$classNameIds)->forceDelete();
        SchoolClass::whereIn('id',$schoolClassIds)->forceDelete();
        Student::whereIn('id',$studentIds)->forceDelete();

        return response()->json(['message' => trans('api.studetn records updated successfully')]);
    }

    /** function update school Teachers  */
    public function updateSchoolTeachrsInfo(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }

        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);

        /** get old ids */
        $employeesIds = SchoolEmployee::where('type', 'teacher')->where('school_id', $school->id)->pluck('id')->toArray();
        $userIds = SchoolEmployee::where('type', 'teacher')->where('school_id', $school->id)->pluck('user_id')->toArray();

        $file = $request->file('file');
        $skipRows = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13]; // Define the rows to skip
        Excel::import(new TeachersImport($file, $skipRows, $request->school_id), $file);

        $teachers = SchoolEmployee::where('type', 'teacher')->where('school_id', $school->id)->pluck('id')->toArray();
        if (count($teachers) <= count($employeesIds)) {
            return response()->json(['message' => trans('api.teachers data not imported check file is a valid file')], 404);
        }

        /** delete school old records before upload new file */
        SchoolEmployee::whereIn('id', $employeesIds)->forceDelete();
        User::whereIn('id', $userIds)->forceDelete();

        /** send email with email caradentials to all employees */
        $school_employees = SchoolEmployee::where('type','teacher')->where('school_id',$school->id)->get();
        foreach ($school_employees as $school_employee) {
            /** create users accounts */
            $temp_password =  'teacher'.mt_rand(100000, 999999);
            $this->createUserAccounts($school_employee->id,$school_employee->name,$school_employee->email,$school_employee->phone,$temp_password);
            //SendEmailJob::dispatch($school_employee); // Dispatch the job for each school employee
        }

        return response()->json(['message' => trans('api.teachers records updated successfully')]);
    }

    public function createUserAccounts($empId,$name,$email,$phone,$temp_password){
        $user = new User();
        $user->name = $name ;
        $user->email = $email ;
        $user->phone = $phone ;
        $user->password = Hash::make($temp_password) ;
        $user->password_reseted = 'no' ;
        $user->status = 'active' ;
        $user->user_type = 'teacher' ;
        $user->verification_code = mt_rand(100000, 999999);
        $user->save();

        /** add user Id to school employee */
        $schoolEmployee = SchoolEmployee::find($empId);
        $schoolEmployee->user_id = $user->id;
        $schoolEmployee->have_account = 'yes';
        $schoolEmployee->save();
    }

    /** function update school Adminstratives  */
    public function updateSchoolAdminstrativesInfo(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);

        return response()->json(['message' => trans('api.data updated successfully')]);
    }

    /** function return current work */
    public function getCurrentWorks(Request $request)
    {
        $currentWorks = CurrentWork::query();
        if($request->type){
            $currentWorks = $currentWorks->where('type',$request->type);
        }

        $currentWorks = CurrentWorkResource::collection($currentWorks->get());
        return response()->json(['currentWorks' => $currentWorks, 'message' => trans('api.current work job titles')]);
    }

    /** function return specializations */
    public function getSpecializations(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }

        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);
        $specializations = SpecializationResource::collection(Specialization::where('school_grade_id',$school->school_grade_id)->where('gender',$school->gender)->get());
        return response()->json(['specializations' => $specializations, 'message' => trans('api.school grade specilizations')]);
    }


    /** function return school classes names */
    public function getClassNames(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        // $school = School::where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);
        $classNames = ClassNameResource::collection(
            ClassName::whereIn('id', function ($query) use ($school) {
                $query->select(DB::raw('MIN(id)'))
                    ->from('class_names')
                    ->where('school_grade_id', $school->school_grade_id)
                    ->where('school_id', $school->id)
                    ->groupBy('name'); // Group by 'name' for distinct results
            })
                ->with(['schoolClasses' => function ($query) use ($school) {
                    $query->where('school_id', $school->id);
                }])
                ->get()
        );
        return response()->json(['classNames' => $classNames, 'message' => trans('api.class names')]);
    }


    /** function return school classes categories */
    public function getClassCategories(Request $request){
        $school = School::findOrFail($request->school_id);

        $classCategories = ClassName::where('school_id', $school->id)
            ->where('name', 'like', "%$request->class_name%")
            ->select('id', 'category')
            ->get()
            ->unique('category')
            ->values();

        return response()->json(['classCategories' => $classCategories, 'message' => trans('api.class categories')]);
    }


    /** function return school classes */
    public function getSchoolClasses(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);

        //$classes = SchoolClassResource::collection(SchoolClass::where('school_id',$school->id)->get());
        $classes = SchoolClass::where('school_id', $school->id)->select('class_number')->distinct()->get();

        return response()->json(['classes' => $classes, 'message' => trans('api.school classes')]);
    }

    /** function return school intsab classes  */
    public function setEntsabSchoolClass(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);

        SchoolClass::where('school_id', $school->id)->update(['entsab' => 'no']);

        /**set entsab classes*/
        $classNameIds = $request->classNameIds;
        foreach ($classNameIds as $classNameId) {
            $class = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $classNameId)->orderBy('id','desc')->first();
            if ($class) {
                $class->entsab = 'yes';
                $class->save();
            }
        }
        return response()->json(['message' => trans('api.school entsab classes is setted')]);
    }

    /** function return school intsab classes */
    public function getEntsabClasses(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $entsabClasses = SchoolClassResource::collection(SchoolClass::where('school_id', $school->id)->where('entsab', 'yes')->get());

        return response()->json(['entsabClasses' => $entsabClasses, 'message' => trans('api.entsab classes')]);
    }

    /** function return school intsab classes students */
    public function getEntsabClassStudents(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);
        $entsabClassIds = SchoolClass::where('school_id', $school->id)->where('entsab', 'yes')->pluck('id')->toArray();

        // Start with the base query
        $data = Student::where('school_id', $school->id)->whereIn('class_id', $entsabClassIds);
        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            return !in_array($key, ['per_page', 'class_id']);
        }, ARRAY_FILTER_USE_KEY);
        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\Student', $filterData);
        }
        // custom filter////
        if ($request->class_id) {
            $data->whereIn('class_id', explode(',', $request->class_id));
        }
        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $filterData = SchoolStudentResource::collection($data->paginate($perPage));

        return response()->json(['filterData' => $filterData, 'message' => trans('api.entsab classes students')]);
    }

    /** functio retrn all schols in the system */
    public function getALlSchools(Request $request)
    {
        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }

        $schools = SchoolResource::collection(School::get());
        // $school = School::findOrFail($request->school_id);
        return response()->json(['schools' => $schools, 'message' => trans('api.all registred schools in system')]);
    }

    /** function return all employees in the school */
    public function getAllSchoolEmployees(Request $request)
    {

        // $token = $request->bearerToken();
        // if ($token) {
        //     $user = JWTAuth::toUser($token);
        // }
        // $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school = School::findOrFail($request->school_id);

        // Start with the base query
        $data = SchoolEmployee::query();
        // Retrieve all query parameters
        $queryParams = $request->query();
        // Apply filters if any
        if (!empty($queryParams)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\SchoolEmployee', $queryParams);
        }

        $data->where('school_id', $school->id)->orderByRaw("FIELD(type, 'adminstrative', 'teacher') ASC");

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $filterData = SchoolEmployeeResource::collection($data->get());

        return response()->json(['schoolEmployees' => $filterData, 'message' => trans('api.all school employees')]);
    }

    /** delete school and its related */
    public function deleteSchool(Request $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $school = School::where('user_id', $user->id)->where('id', $request->school_id)->firstOrFail();
        $school->delete();
        return response()->json(['message' => trans('api.record deleted successflly')]);
    }

    public function get_schoolClass_students($school_class_id)
    {
        $students  = Student::where('class_id', $school_class_id)->select('id', 'name')->get();
        return response()->json([
            'students' => $students
        ]);
    }

    public function getSchoolFacilitiesJson($schoolId)
    {
        $roomTypes = [
            'n_class' => trans('api.Class'),
            'n_team_rooms' => trans('api.Team Room'),
            'n_management_rooms' => trans('api.Management Room'),
            'n_computer_laps' => trans('api.Computer Lab'),
            'n_science_laps' => trans('api.Science Lab'),
            'n_learning_resources' => trans('api.Learning Resource'),
            'n_activity_rooms' => trans('api.Activity Room'),
            'n_meeting_rooms' => trans('api.Meeting Room'),
        ];

        $result = [];
        $school = School::findOrFail($schoolId);
        $schoolFacilities = SchoolFacilitie::where('school_id', $school->id)->firstOrFail();
        foreach ($roomTypes as $key => $name) {
            $count = $schoolFacilities->$key ?? 0;

            if ($count > 0) {
                for ($i = 1; $i <= $count; $i++) {
                    $result[] = "$name " . $i;
                }
            }
        }

        return response()->json([
            'locations' => $result,
            'message' => trans('api.school locations')
        ]);
    }
    public function get_classes_for_category(School $school, ClassName $categroy)
    {

        $classes = SchoolClass::where('class_name_id', $categroy->id)->where('school_id', $school->id)->get()->select('id', 'class_number');
        return response()->json($classes);
    }
    public function get_subject_for_class(School $school, SchoolClass $class)
    {
        $teacher_schedules = TeacherSchedule::with('schedule_data.subject')->where('school_id', $school->id)->where('school_class_id', $class->id)->get()->pluck('id')->toArray();
        $teacher_schedule_data = TeacherScheduleData::with('subject')->whereIn('teacher_schedule_id', $teacher_schedules)->get()
            ->map(function ($teacher_schedule_sub) {
                return [
                    'subject_id' => $teacher_schedule_sub?->subject->id,
                    'subject_name' => $teacher_schedule_sub?->subject->name,
                ];
            })->unique('subject_id')->values();
        return response()->json($teacher_schedule_data);
    }
    public function get_teacher_for_subject(School $school, Subject $subject)
    {
        $teacher_schedules = TeacherSchedule::with('schedule_data.subject')
            ->where('school_id', $school->id)->get()->pluck('id')->toArray();
        $teacher_schedule_data = TeacherScheduleData::with('employee')
            ->where('subject_id', $subject->id)
            ->whereIn('teacher_schedule_id', $teacher_schedules)->get()
            ->map(function ($teacher_id) {
                return [
                    'employee_id' => $teacher_id?->employee->id,
                    'employee_name' => $teacher_id?->employee->name
                ];
            })->unique('employee_id');
        return response()->json($teacher_schedule_data);
    }
    public function get_current_semester(School $school)
    {
        $current_semester =  Helper::getCurrentSchoolSemester();
        $school_year = $school->current_school_year;
        return response()->json(['current_semester_id' => $current_semester->id,'current_semester' => $current_semester->name, 'school_year' => $school_year]);
    }

    /** function return response of semester weeks */
    public  function getWeeks(){
        $weeks = WeekResource::collection(Week::all());
        return response()->json(['weeks'=>$weeks,'message' => trans('api.list of semester weeks')]);
    }

    public function getCurrentWorkEmployees($schoolId,$currentWorkId){
        $school = School::where('id',$schoolId)->firstOrFail();
        $schoolEmployees = SchoolEmployee::where('school_id', $schoolId)->where('current_work_id',$currentWorkId)->get();

        if ($schoolEmployees->isEmpty()) {
            return [];
        }

        $data = $schoolEmployees->map(function ($schoolEmployee) {
            return[
                'id' => $schoolEmployee->id,
                'name' => $schoolEmployee->name,
                'ssn' => $schoolEmployee?->ssn,
                'specialization' =>  $schoolEmployee->specialization?->title,
            ];
        })->all();

        return response()->json(['school_employees'=>$data,'message' => trans('api.all current work employees')]);
    }


    public function saveYearCalendar(Request $request) {
        try {
            $startDate = Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            $endDate = Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
        } catch (\Exception $e) {
            return response()->json(['message' => trans('api.invalid_date_format')], 400);
        }

        // Validate the dates
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date_format:d/m/Y',
            'end_date' => 'required|date_format:d/m/Y|after_or_equal:start_date',
        ]);

        $existingCalendar = DB::table('semester_week_days')->where('date', $request->start_date)->first();

        if ($existingCalendar) {
            return response()->json(['message' => trans('api.school Year Calendar already exists')], 400);
        }

        Helper::saveSchoolYearCalendar($request->start_date, $request->end_date);

        return response()->json(['message' => trans('api.school Year Calendar added successfully')]);
    }


}
