<?php

namespace App\Http\Controllers\Api\ManagerDashboard\TeamAndCommitteeMeetings;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\ManagerDashboard\TeamAndCommitteeMeetings\TeamRequest;
use App\Http\Requests\ManagerDashboard\TeamAndCommitteeMeetings\UpdateTeamRequest;
use Illuminate\Http\Request;
use App\Http\Resources\ManagerDashboard\TeamAndCommitteeMeetings\TeamResource;
use App\Models\AdminDashboard\School\CurrentWork;
use App\Models\AdminDashboard\School\School;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Team;
use App\Traits\LogsActivity;

class TeamController extends Controller
{
    use LogsActivity;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:teams']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request, School $school)
    {
        // Start with the base query
        $data = Team::with(['meetings' => function ($query) use ($school) {
            // Filter meetings by school_id
            $query->where('school_id', $school->id);
        }]);

        // Retrieve all query parameters
        $queryParams = $request->query();
        // Apply filters if any
        if (!empty($queryParams)) {
            $data = Helper::filterData('App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Team', $queryParams);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        $filterData = TeamResource::collection($data->where('school_id', Null)->orWhere('school_id', $school->id)->paginate($perPage));
        return response()->json([
            'teams' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored 
        by using ID
    */
    public function getById(Team $team, School $school)
    {
        // Default assignment works
        $assignmentWorks = [];

        // Define assignment works based on the form number
        if ($team->assignment_form_number == 28) {
            $assignmentWorks = ['رئيساً للفريق', 'عضو', 'عضو', 'عضو', 'عضو', 'مقرر الفريق'];
        } elseif ($team->assignment_form_number == 29) {
            $assignmentWorks = ['رئيساً للفريق', 'عضو', 'عضو', 'عضو', 'مقرر الفريق'];
        } elseif ($team->assignment_form_number == 34) {
            $assignmentWorks = ['رئيساً للفريق', 'عضو', 'عضو', 'عضو', 'عضو', 'مقرر الفريق'];
        } else {
            $assignmentWorks = ['رئيساً للفريق', 'عضو', 'عضو', 'عضو', 'عضو', 'عضو'];
        }

        // Define work IDs for each form number
        $workIds = [
            28 => [7, 11, 10, 2, 2, 3],
            29 => [8, 11, 15, 2, 9],
            34 => [8, 11, 10, 2, 2, 9],
            35 => []
        ];

        // Check if assignment_form_number is 35 or null
        if ($team->assignment_form_number == 35 || $team->assignment_form_number == null) {
            $currentWork = [];
        } else {
            // Get the current work IDs for the specific form number
            $currentWorkIds = $workIds[$team->assignment_form_number] ?? [];

            // Fetch current work records based on the currentWorkIds
            $currentWork = CurrentWork::with(['employees' => function ($query) use ($school) {
                $query->where('school_id', $school->id);
            }])
                ->select('id', 'job_title')
                ->whereIn('id', $currentWorkIds)
                ->get();

            // Reorder and duplicate the works based on the order in work_ids
            $currentWork = collect($currentWorkIds)->map(function ($workId) use ($currentWork) {
                return $currentWork->firstWhere('id', $workId);
            });
        }


        // Define the default work ids to fetch employees if current work has no employees
        $defaultWorkIds = [1, 6, 7, 8];

        // Loop through currentWorks and check if any work has no employees
        if($currentWork)
        {
            $currentWork->each(function ($work) use ($defaultWorkIds) {
                if ($work->employees->isEmpty()) {
                    // If no employees for the current work, fetch from defaultWorkIds
                    $defaultWork = CurrentWork::with(['employees' => function ($query) {
                        $query->where('school_id', auth()->user()->school_id);
                    }])
                        ->select('id', 'job_title')
                        ->whereIn('id', $defaultWorkIds)
                        ->first();
    
                    // Replace empty employees with the fetched employees from the default work
                    if ($defaultWork && !$work->employees->isNotEmpty()) {
                        $work->employees = $defaultWork->employees;
                    }
                }
            });
        }
      

        // Load meetings for the team
        $team->load(['meetings' => function ($query) use ($school) {
            $query->where('school_id', $school->id);
        }]);

        // Pass the current work, assignment works, and flag to TeamResource
        $teamResource = new TeamResource($team, $currentWork, $assignmentWorks, true);

        return response()->json([
            'teams' => $teamResource,
            'message' => trans('api.team details')
        ], 200);
    }


    /* this method will stord data to DB Model*/
    public function store(TeamRequest $request)
    {

        $data = $request->validated();

        $team = new Team();
        $team->school_id =  $data['school_id'];
        $team->name = $data['name'];
        $team->form_number = $data['form_number'] ?? 35;
        $team->assignment_form_number = $data['assignment_form_number'] ?? 35;
        $team->status =  $data['status'] ?? 1;
        $team->save();
        // ------------------------------- School ID -----------------------
        $school_id =  $data['school_id'];
        $this->ManagerlogActivity('Team', 'store', $school_id);

        return response()->json(['message' => trans('api.record stored successflly')], 200);
    }

    /* this method will update specific Model recored 
        by using ID
    */
    public function update(UpdateTeamRequest $request, Team $team, School $school)
    {

        $data =  $request->validated();

        $team->update($data);

        // ------------------------------- School ID -----------------------
        $school_id =  $school->id;
        $this->ManagerlogActivity('Team', 'update', $school_id);

        return response()->json(['message' => trans('api.record updated successflly')], 200);
    }

    /* this method will change activation status for
       specific Model recored by using ID
    */
    public function changeStatus(Request $request, Team $team, School $school)
    {
        $team->status = $request->status;
        $team->save();
        return response()->json(['message' => trans('api.status changed successflly')], 200);
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive(Team $team, School $school)
    {

        $team->delete();
        $this->ManagerlogActivity('Team', 'archive',  $school->id);


        return response()->json(['message' => trans('api.record archived successflly')], 200);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function destroy(Team $team, School $school)
    {
        $team->forceDelete();
        $this->ManagerlogActivity('Team', 'delete',  $school->id);
        return response()->json(['message' => trans('api.record deleted successflly')], 200);
    }

    /* this method will restore specific Model recored 
        by using ID
    */
    public function restore(Team $team, School $school)
    {
        $team->restore();
        $this->ManagerlogActivity('Team', 'restore', $school->id);
        return response()->json(['message' => trans('api.record restored successflly')], 200);
    }
}
