<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Tables\WaitingPeriodsSchedule;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\ManagerDashboard\waitingPeriodsSchedule\WaitingPeriodsScheduleDataRequest;
use App\Http\Requests\ManagerDashboard\waitingPeriodsSchedule\WaitingPeriodsScheduleRequest;
use App\Http\Resources\ManagerDashboard\WaitingPeriodsSchedule\WaitingPeriodsScheduleResource;
use App\Jobs\SendSMSNotification;
use App\Models\AdminDashboard\School\Period;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\SchoolEmployeeAttendance;
use App\Models\AdminDashboard\School\SemesterWeekDay;
use App\Models\ManagerDashboard\NotificationSettings\NotificationSchoolSetting;
use App\Models\ManagerDashboard\TeacherSchedule;
use App\Models\ManagerDashboard\Tables\WaitingPeriodsSchedule\WaitingPeriodsSchedule;
use App\Models\ManagerDashboard\Tables\WaitingPeriodsSchedule\WaitingPeriodsScheduleData;
use App\Models\Namazeg;
use App\Traits\LogsActivity;
use App\Traits\NamazegTrait;
use Carbon\Carbon;
use JWTAuth;

class WaitingPeriodsScheduleController extends Controller
{
    use LogsActivity, NamazegTrait;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        // $this->middleware(['auth:api', 'jwt.verify','permission:waitingPeriodsSchedules']);
    }

    /*
        -this method will return all Model recoreds
    */

    public function getAll(Request $request)
    {
        $data = WaitingPeriodsSchedule::where('school_id', $request->school_id);
        $teacher_ids = WaitingPeriodsSchedule::where('school_id', $request->school_id)->pluck('school_employee_id')->toArray();

        $queryParams = $request->query();
        // Apply filters if any
        if (isset($queryParams['teacher_name'])) {
            $name = $queryParams['teacher_name'];
            $ids = SchoolEmployee::whereIn('id', $teacher_ids)->where('name', 'like', "%$name%")->pluck('id')->toArray();
            $data->whereIn('school_employee_id', $ids);
        }

        if (isset($queryParams['schedules_to_view'])) {
            if ($queryParams['schedules_to_view'] == 'today') {
                $date = Carbon::now()->format('d/m/Y');
                $data->where('creation_date', 'like', "%$date%");
            } elseif ($queryParams['schedules_to_view'] == 'other') {
                $date = Carbon::now()->format('d/m/Y');
                $data->where('creation_date', '!=', "$date");
            }
        }

        if (isset($queryParams['creation_date']) && isset($queryParams['creation_date_type'])) {
            $date = $queryParams['creation_date'];
            if ($queryParams['creation_date_type'] == "hijri") {
                $date = Helper::getCoptic($queryParams['creation_date']);
            }
            $data->where('creation_date', 'like', "%$date%");
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $paginatedData = $data->paginate($perPage);

        $groupedData = WaitingPeriodsScheduleResource::collection($paginatedData);

        return response()->json([
            'waitingPeriodsSchedules' => $groupedData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored 
        by using ID
    */
    public function getById($schedule_id)
    {
        $schedule = WaitingPeriodsSchedule::findOrFail($schedule_id);
        $waitingPeriodsSchedule = new WaitingPeriodsScheduleResource($schedule);
        return response()->json(['waitingPeriodsSchedule' => $waitingPeriodsSchedule, 'message' => trans('api.waiting Periods Schedule details')], 200);
    }

    /* this method will get Absent Teachers data in specific date*/
    public function getAbsentTeachers(Request $request)
    {
        // Validate required parameters
        if (!$request->school_id || !$request->creation_date || !$request->creation_date_type) {
            return response()->json(['error' => 'Missing required parameters'], 400);
        }
        if ($request->creation_date_type == "hijri") {
            $creation_date = Helper::getCoptic($request->creation_date);
        } else {
            try {
                $creation_date = Carbon::createFromFormat('d/m/Y', $request->creation_date)->format('d/m/Y');
            } catch (\Exception $e) {
                return response()->json(['error' => 'Invalid date format. Please use d/m/Y.'], 400);
            }
        }
        $absentTeachers_ids = SchoolEmployeeAttendance::where('school_id', $request->school_id)->where('status', 'absent')->where('date', 'like', "%$creation_date%")->pluck('school_employee_id')->toArray();
        $absentTeachers = SchoolEmployee::whereIn('id', $absentTeachers_ids)->get();
        return response()->json(['absentTeachers' => $absentTeachers], 200);
    }

    /* this method will stord data to DB Model*/
    public function storeSchedule(WaitingPeriodsScheduleRequest $request)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        $creation_date = Carbon::createFromFormat('Y-m-d', $request->creation_date)->format('d/m/Y');
        $SemesterWeekDay = SemesterWeekDay::where('semester_id', $request->semester->id)->where('date', 'like', "$creation_date")->firstOrFail();
        $absent_periods_count = TeacherSchedule::join('teacher_schedul_data as schedule_data', 'schedule_data.teacher_schedule_id', 'teacher_schedules.id')
            ->where('school_id', $request->school_id)
            ->where('semester_id', $request->semester->id)
            ->where('day_id', $SemesterWeekDay->day_id)
            ->where('school_employee_id', $request->school_employee_id)
            ->pluck('period_id')
            ->count();

        $namozeg = Namazeg::where('form_number', 51)->first();
        $namozeg_version_id = $this->storeNamazegVersion(
            $namozeg->id,
            $namozeg->internal_extenral,
            $request->school_id
        )->id;

        $schedule = WaitingPeriodsSchedule::create([
            'school_employee_id' => $request->school_employee_id,
            'school_id' => $request->school_id,
            'user_id' => $user->id,
            'semester_id' => $request->semester->id,
            'namazeg_version_id' => $namozeg_version_id,
            'day_id' => $SemesterWeekDay->day_id,
            'absent_periods_count' => $absent_periods_count,
            'creation_date' => $creation_date,
        ]);
        // ------------------------------- School ID -----------------------
        $school_id = $schedule->school_id;
        $this->ManagerlogActivity('WaitingPeriodsSchedule', 'storeSchedule', $school_id);

        return response()->json(['schedule_id' => $schedule->id, 'periods_count' => $absent_periods_count, 'message' => trans('api.record stored successflly')]);
    }

    /* this method will get Attended Teachers data in specific date*/
    public function getAttendedTeachers(Request $request)
    {
        if ($request->creation_date_type == "hijri") {
            $creation_date = Helper::getCoptic($request->creation_date);
        } else {
            try {
                $creation_date = Carbon::createFromFormat('d/m/Y', $request->creation_date)->format('d/m/Y');
            } catch (\Exception $e) {
                return response()->json(['error' => 'Invalid date format. Please use d/m/Y.'], 400);
            }
        }
        
        $semester = Helper::getCurrentSchoolSemester($request->creation_date);

        $SemesterWeekDay = SemesterWeekDay::where('semester_id', $semester->id)->where('date', 'like', "$creation_date")->firstOrFail();
        $periods = Period::pluck('id')->toArray();

        foreach ($periods as $period) {
            $teacherscheduleIds = TeacherSchedule::join('teacher_schedul_data as schedule_data', 'schedule_data.teacher_schedule_id', 'teacher_schedules.id')
                ->where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('period_id', $period)
                ->where('day_id', $SemesterWeekDay->day_id)
                ->groupBy('school_employee_id')
                ->pluck('school_employee_id')
                ->toArray();

            $adminstrativesIds = SchoolEmployee::where('school_id', $request->school_id)
                ->where('type', 'adminstrative')
                ->whereNotNull('user_id')
                ->pluck('id')
                ->toArray();

            $ids = array_merge($teacherscheduleIds, $adminstrativesIds);

            $attendedEmployeesIds = SchoolEmployeeAttendance::where('school_id', $request->school_id)
                ->where('status', 'attended')
                ->where('date', 'like', "%$creation_date%")
                ->whereNotIn('school_employee_id', $ids)
                ->pluck('school_employee_id')
                ->toArray();

            $attendedEmployees = SchoolEmployee::where('school_id', $request->school_id)->whereIn('id', $attendedEmployeesIds)->get();
            $data[$period] = $attendedEmployees;
        }

        return response()->json(['attendedEmployees' => $data], 200);
    }

    /* this method will stord data to DB Model*/
    public function storeOrUpdateScheduleData(WaitingPeriodsScheduleDataRequest $request, WaitingPeriodsSchedule $waitingPeriodsSchedule)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }
        WaitingPeriodsScheduleData::where('schedule_id', $waitingPeriodsSchedule->id)->delete();
        foreach ($request->data as $record) {
            $school_class = SchoolClass::where('school_id', $waitingPeriodsSchedule->school_id)->where('class_name_id', $record['class_name_id'])->where('class_number', $record['class_number'])->first();
            if (!$school_class) return response()->json(['errors' => trans('api.class_name_id or class_number not valid')]);
            $schedual = WaitingPeriodsScheduleData::create([
                'schedule_id' => $waitingPeriodsSchedule->id,
                'school_employee_id' => $record['school_employee_id'],
                'school_class_id' => $school_class->id,
                'subject_id' => $record['subject_id'],
                'period_id' => $record['period_id'],
            ]);
            $message_text = "لديك تأمين حصص انتظار مجدولة هذااليوم. نرجو الالتزام بالحضور فى الموعد المحدد.";
            $date = $waitingPeriodsSchedule->creation_date;
            $sendAt = Carbon::createFromFormat('d/m/Y', $date)->setTime(5, 0, 0);
            SendSMSNotification::dispatch($schedual->employee->phone, $message_text, $user->manager_id)->delay($sendAt);
        }
        // ------------------------------- School ID -----------------------
        $school_id = $waitingPeriodsSchedule->school_id;
        $this->ManagerlogActivity('WaitingPeriodsScheduleData', 'storeOrUpdateScheduleData', $school_id);

        return response()->json(['message' => trans('api.record stored successflly')]);
    }

    public function updateSchedule(WaitingPeriodsScheduleRequest $request, WaitingPeriodsSchedule $waitingPeriodsSchedule)
    {
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }

        $creation_date = Carbon::createFromFormat('Y-m-d', $request->creation_date)->format('d/m/Y');
        $SemesterWeekDay = SemesterWeekDay::where('semester_id', $request->semester_id)->where('date', 'like', "$creation_date")->firstOrFail();

        $absent_periods_count = TeacherSchedule::join('teacher_schedul_data as schedule_data', 'schedule_data.teacher_schedule_id', 'teacher_schedules.id')
            ->where('school_id', $request->school_id)
            ->where('semester_id', $request->semester_id)
            ->where('day_id', $SemesterWeekDay->day_id)
            ->where('school_employee_id', $request->school_employee_id)
            ->pluck('period_id')
            ->count();

        $waitingPeriodsSchedule->update([
            'school_employee_id' => $request->school_employee_id,
            'school_id' => $request->school_id,
            'user_id' => $user->id,
            'semester_id' => $request->semester_id,
            'day_id' => $SemesterWeekDay->day_id,
            'absent_periods_count' => $absent_periods_count,
            'creation_date' => $creation_date,
        ]);
        // ------------------------------- School ID -----------------------
        $school_id = $waitingPeriodsSchedule->school_id;
        $this->ManagerlogActivity('WaitingPeriodsSchedule', 'updateSchedule', $school_id);

        return response()->json(['schedule_id' => $waitingPeriodsSchedule->id, 'periods_count' => $absent_periods_count, 'message' => trans('api.record stored successflly')]);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function delete($id)
    {
        $schedul = WaitingPeriodsSchedule::findOrFail($id);
        $school_id = $schedul->school_id;
        $schedul->forceDelete();
        // ------------------------------- School ID -----------------------
        $this->ManagerlogActivity('WaitingPeriodsScheduleData', 'delete', $school_id);
        return response()->json(['message' => trans('api.record deleted successflly')]);
    }
}
