<?php

namespace App\Http\Controllers\Api\ManagerDashboard\Tables\Exams;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ManagerDashboard\Tables\Exams\TableExamResource;
use App\Http\Resources\School\SchoolStudentResource;
use App\Models\AdminDashboard\School\ClassName;
use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\SchoolSemester;
use App\Models\AdminDashboard\School\SemesterWeekDay;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\Tables\Exams\TableExam;
use App\Models\ManagerDashboard\Tables\Exams\TableExamDay;
use App\Models\ManagerDashboard\Tables\Exams\TableExamDayExam;
use App\Models\ManagerDashboard\Tables\Exams\TableExamNamazeg;
use App\Models\ManagerDashboard\Tables\Exams\TableExamInstruction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use JWTAuth;
use Illuminate\Support\Facades\Validator;
use App\Models\Namazeg;
use App\Traits\LogsActivity;
use App\Traits\NamazegTrait;
use App\Traits\SmsTrait;


class TableExamController extends Controller
{
    use LogsActivity, NamazegTrait , SmsTrait;

    public function __construct()
    {
        //$this->middleware(['auth:api','permission:table-exams']);
    }

    /** function get all records */
    public function getAll(Request $request){
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }

        // Validate that the school exists for this user
        $school = School::where('user_id', $user->id)
            ->where('id', $request->school_id)
            ->firstOrFail();


        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            return !in_array($key, ['per_page', 'semester_id']);
        }, ARRAY_FILTER_USE_KEY);
        // Apply filters if any
        if (!empty($filterData)) {
            $periodicTableExams =Helper::filterData('App\Models\ManagerDashboard\Tables\Exams\TableExam', $filterData);
            $finalWrittenTableExams = Helper::filterData('App\Models\ManagerDashboard\Tables\Exams\TableExam', $filterData);
            $finalPracticalTableExams =Helper::filterData('App\Models\ManagerDashboard\Tables\Exams\TableExam', $filterData);
        }

        $periodicTableExams = TableExam::where('school_id',$school->id)->where('exam_type','periodic');
        $finalWrittenTableExams = TableExam::where('school_id',$school->id)->where('exam_type','final')->where('exam_category','written');
        $finalPracticalTableExams = TableExam::where('school_id',$school->id)->where('exam_type','final')->where('exam_category','verbal')->orWhere('exam_category','practical');

        // custom filter////
        if ($request->semester_id) {
            $periodicTableExams->whereIn('semester_id', explode(',',$request->semester_id));
            $finalWrittenTableExams->whereIn('semester_id', explode(',',$request->semester_id));
            $finalPracticalTableExams->whereIn('semester_id', explode(',',$request->semester_id));
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $periodicTableExams = TableExamResource::collection($periodicTableExams->paginate($perPage));
        $finalWrittenTableExams = TableExamResource::collection($finalWrittenTableExams->paginate($perPage));
        $finalPracticalTableExams = TableExamResource::collection($finalPracticalTableExams->paginate($perPage));

        return response()->json(['periodicTableExams'=>$periodicTableExams,
            'finalWrittenTableExams'=>$finalWrittenTableExams,
            'finalPracticalTableExams'=>$finalPracticalTableExams,
            'message' => trans('api.all table exams records')]);
    }


    /** function get record byId */
    public function getById(Request $request){
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }

        // Validate that the school exists for this user
        $school = School::where('user_id', $user->id)
            ->where('id', $request->school_id)
            ->firstOrFail();
        $tableExam = new TableExamResource(TableExam::findOrFail($request->table_exam_id));

        return response()->json(['tableExam'=>$tableExam,'message' => trans('api.table exams details')]);
    }

    /** function delete record byId */
    public function destory(Request $request){
        $token = $request->bearerToken();
        if ($token) {
            $user = JWTAuth::toUser($token);
        }

        // Validate that the school exists for this user
        $school = School::where('user_id', $user->id)
            ->where('id', $request->school_id)
            ->firstOrFail();
        $tableExam = TableExam::findOrFail($request->table_exam_id);
        $tableExam->delete();

        return response()->json(['message' => trans('api.record deleted successfully')]);
    }


    public function store(Request $request, $step){
        // Authenticate the user
        $user = $this->getUserFromToken($request->bearerToken());

        // Validate school
        $school = $this->validateSchool($user->id, $request->school_id);

        switch ($step) {
            case 1:
                return $this->handleCreateExamTable($request, $school);

            case 2:
                return $this->handleUpdateExamDays($request);

            default:
                return response()->json(['message' => trans('api.invalid step')], 400);
        }
    }

    /**
     * Get user from token.
     */
    protected function getUserFromToken($token)
    {
        if ($token) {
            return JWTAuth::toUser($token);
        }
        throw new UnauthorizedHttpException('Token not provided');
    }

    /**
     * Validate the school for the user.
     */
    protected function validateSchool($userId, $schoolId)
    {
        return School::where('user_id', $userId)
            ->where('id', $schoolId)
            ->firstOrFail();
    }

    /**
     * Handle the creation of an exam table.
     */
    protected function handleCreateExamTable($request, $school)
    {
        $examsStartDay = SemesterWeekDay::where('date', $request->start_date)->first();

        if ($examsStartDay) {
            return $this->createExamTable($request, $school, $examsStartDay);
        } elseif (!$examsStartDay && $request->round === 'second') {

            return $this->createSecondRoundExamTable($request, $school);
        }

        return response()->json(['message' => trans('api.invalid start date')], 400);
    }

    /**
     * Create exam table for the first round.
     */
    protected function createExamTable($request, $school, $examsStartDay){
        $schoolSemester = Helper::getCurrentSchoolSemester();
        if (!$schoolSemester || $schoolSemester->id !== $examsStartDay->semester_id) {
            return response()->json(['message' => trans('api.invalid start date or no semester found')], 400);
        }

        if (!Helper::checkExamStartDate($examsStartDay->date, $schoolSemester->id)) {
            return response()->json(['message' => trans('api.cant create exams table , invalid start date')], 400);
        }

        $data = $request->except('class_name_id','class_id','class_name_categories');
        $data['semester_id'] =$schoolSemester->id;

        if (isset($request['start_date_type']) && isset($request['start_date']) ) {
            $start_date = $request['start_date'];

            if ($request['start_date_type'] == "hijri") {
                $start_date = Helper::getCoptic($request['start_date']);
            }
        }
        $data['start_date'] =$start_date;
        $data['start_date_type'] ='coptic';

        $tableExam = TableExam::create($data);
        $examDays = $this->getExamDays($request);

        $this->createTableExamDays($request, $tableExam, $examDays, $school);

        if($request->exam_type == 'periodic'){
            $this->smsMonthlyExames($request->school_id,$request->class_id,$request->start_date);
        }

        // if($request->exam_type == 'final'){
        //     if($request->table_type == 'standard'){
        //         $this->smsStandredFinalExames($request->school_id,$request->class_name_id,$request->start_date);
        //     }
        //     if($request->table_type == 'different'){
        //         $this->smsDifferentFinalExames($request->school_id,$request->class_id,$request->start_date);
        //     }
        // }

        return response()->json(['tableExam' => new TableExamResource($tableExam), 'message' => trans('api.exams table created successfully')]);
    }

    /**
     * Create exam table for the second round.
     */
    protected function createSecondRoundExamTable($request, $school)
    {
        $tableExam = TableExam::create($request->except('class_name_id', 'class_id'));

        $examDays = Helper::getWorkingDays($request->start_date, 5);

        $this->createTableExamDays($request, $tableExam, $examDays, $school);

        return response()->json(['tableExam' => new TableExamResource($tableExam), 'message' => trans('api.exams table created successfully')]);
    }

    /**
     * Get exam days.
     */
    protected function getExamDays($request)
    {
        return SemesterWeekDay::where('date', $request->start_date)
            ->orWhere(function ($query) use ($request) {
                $examsStartDay = SemesterWeekDay::where('date', $request->start_date)->first();
                $query->where('id', '>', $examsStartDay->id);
            })
            ->orderBy('id', 'asc')
            ->take(5)
            ->get();
    }

    /**
     * Create exam days for the table exam.
     */
    protected function createTableExamDays($request, $tableExam, $examDays, $school)
    {
        if ($tableExam->exam_type === 'periodic') {
            $this->createExamDaysForPeriodic($request, $tableExam, $examDays);
        } elseif ($tableExam->exam_type === 'final') {
            $this->createExamDaysForFinal($request, $tableExam, $examDays, $school);
        }
    }

    /**
     * Create exam days for periodic exams.
     */
    protected function createExamDaysForPeriodic($request, $tableExam, $examDays)
    {
        foreach ($examDays as $examDay) {
            TableExamDay::create([
                'table_exam_id' => $tableExam->id,
                'class_name_id' => $request->class_name_id[0],
                'class_id' => $request->class_id[0],
                'semester_week_day_id' => $examDay->id,
                'n_exams' => 0,
            ]);
        }

        $this->handelPeriodicExamNamazeg($tableExam->id);
    }

    /**
     * Create exam days for final exams.
     */
    protected function createExamDaysForFinal($request, $tableExam, $examDays, $school)
    {
        $school = School::find($request->school_id);
        if ($tableExam->table_type === 'standard') {
            if($school->school_grade_id == 3){
                $this->createExamDaysForStandardFinalSecondarySchool($request, $tableExam, $examDays,$school);
            }else{
                $this->createExamDaysForStandardFinal($request, $tableExam, $examDays,$school);
            }

        } elseif ($tableExam->table_type === 'different') {
            $this->createExamDaysForDifferentFinal($request, $tableExam, $examDays, $school);
        }
    }

    /**
     * Create exam days for standard final exams.
     */
    protected function createExamDaysForStandardFinal($request, $tableExam, $examDays,$school){
        /** convert array to object*/
        $classNameIds = (object)$request->class_name_id;
        foreach ($classNameIds as $classNameId) {
            foreach ($examDays as $examDay) {
                TableExamDay::create([
                    'table_exam_id' => $tableExam->id,
                    'class_name_id' => $classNameId,
                    'class_id' => NULL,
                    'semester_week_day_id' => ($tableExam->round == 'second')? Null :$examDay->id ,
                    'date' => ($tableExam->round == 'second')? $examDay:NULL,
                    'date_type' =>($tableExam->round == 'second')? 'coptic' :Null,
                    'n_exams' => 0,
                ]);
            }
        }

        $this->handelFinalStandardExamNamazeg($tableExam->id);
    }


    protected function createExamDaysForStandardFinalSecondarySchool($request, $tableExam, $examDays,$school){

        if($request->class_name_categories){
            $classNameId = $request->class_name_id;
            $className= ClassName::find($classNameId[0]);

            $classNameIds =  ClassName::where('school_id',$request->school_id)->where('name',$className->name)->whereIn('category',$request->class_name_categories)->pluck('id')->toArray();

            foreach ($classNameIds as $classNameId) {
                $className = ClassName::find($classNameId);
                foreach ($examDays as $examDay) {
                    TableExamDay::create([
                        'table_exam_id' => $tableExam->id,
                        'class_name_id' => $classNameId,
                        'class_name_category' => $className->category,
                        'class_id' => NULL,
                        'semester_week_day_id' => ($tableExam->round == 'second')? Null :$examDay->id ,
                        'date' => ($tableExam->round == 'second')? $examDay:NULL,
                        'date_type' =>($tableExam->round == 'second')? 'coptic' :Null,
                        'n_exams' => 0,
                    ]);
                }
            }

        }else{
            /** convert array to object*/
            $classNameIds = (object)$request->class_name_id;
            foreach ($classNameIds as $classNameId) {
                $className = ClassName::find($classNameId);
                foreach ($examDays as $examDay) {
                    TableExamDay::create([
                        'table_exam_id' => $tableExam->id,
                        'class_name_id' => $classNameId,
                        'class_name_category' => NULL,
                        'class_id' => NULL,
                        'semester_week_day_id' => ($tableExam->round == 'second')? Null :$examDay->id ,
                        'date' => ($tableExam->round == 'second')? $examDay:NULL,
                        'date_type' =>($tableExam->round == 'second')? 'coptic' :Null,
                        'n_exams' => 0,
                    ]);
                }
            }
        }


        $this->handelFinalStandardExamNamazeg($tableExam->id);
    }

    /**
     * Create exam days for different final exams.
     */
    protected function createExamDaysForDifferentFinal($request, $tableExam, $examDays, $school){

        $classes = SchoolClass::where('school_id', $school->id)->whereIn('class_name_id', $request->class_name_id)->get();

        foreach ($classes as $class) {
            foreach ($examDays as $examDay) {
                TableExamDay::create([
                    'table_exam_id' => $tableExam->id,
                    'class_name_id' => $request->class_name_id[0],
                    'class_id' => $class->id,
                    'semester_week_day_id' => ($tableExam->round == 'second')? Null :$examDay->id ,
                    'date' => ($tableExam->round == 'second')? $examDay:NULL,
                    'date_type' =>($tableExam->round == 'second')? 'coptic' :Null,
                    'n_exams' => 0,
                ]);
            }
        }

        $this->handelFinalDiffrentExamNamazeg($tableExam->id);
    }

    /**
     * Handle updating exam days with subjects.
     */
    protected function handleUpdateExamDays($request){
        $tableExam = TableExam::find($request->table_exam_id);
        $examDays = TableExamDay::where('table_exam_id', $request->table_exam_id)->get();

        if($tableExam->table_type == 'different' && $request->class_id){
            $examDays = TableExamDay::where('table_exam_id', $request->table_exam_id)->where('class_id',$request->class_id)->get();
        }

        if($tableExam->table_type == 'standard' && $request->class_name_id){
            if($request->class_name_category){
                $examDays = TableExamDay::where('table_exam_id', $request->table_exam_id)->where('class_name_category',$request->class_name_category)->where('class_name_id',$request->class_name_id)->get();
            }else{
                $examDays = TableExamDay::where('table_exam_id', $request->table_exam_id)->where('class_name_id',$request->class_name_id)->get();
            }
        }

        foreach($request->days_exams as $dayExam){
            /** conver array to bject*/
            $dayExamObject = (object)$dayExam;

            $dayExam = TableExamDay::find($dayExamObject->exam_day_id);
            $dayExam->update([
                'n_exams' => $dayExamObject->n_exams,
                'leave_time'=>$dayExamObject->leave_time,
            ]);

            /** remove old exams ans insert new**/
            TableExamDayExam::where('table_exam_day_id',$dayExam->id)->delete();

            foreach($dayExamObject->exams as $exam){
                /** conver array to bject*/
                $exam = (object)$exam;

                $examDayExam = new TableExamDayExam();
                $examDayExam->table_exam_day_id =$dayExam->id;
                $examDayExam->subject_id = $exam->subject_id;
                $examDayExam->start_time = $exam->start_time;
                $examDayExam->end_time = $exam->end_time;
                $examDayExam->save();
            }
        }

        return response()->json(['message' => trans('api.table exam days updated successfully')]);
    }

    public function updateExamInstructions(Request $request){
        $examInstructions = TableExamInstruction::find(1);
        if($examInstructions){
            $examInstructions->text = $request->text;
            $examInstructions->save();
            return response()->json(['message' => trans('api.exams instructions updated successfully')]);
        }

        $examInstructions = new TableExamInstruction();
        $examInstructions->text = $request->text;
        $examInstructions->save();
        return response()->json(['message' => trans('api.exams instructions updated successfully')]);
    }

    public function showExamInstructions(){
        $examInstructions = TableExamInstruction::findOrFail(1);
        return response()->json(['message' => trans('api.exams instructions updated successfully')]);
    }


    private function handelPeriodicExamNamazeg($tableExamId){
        $tableExam = TableExam::find($tableExamId);
        /** create or update  namozg version for this table*/
        $namozeg = Namazeg::where('form_number', 233)->first();
        $namazeg_version_id = $this->storeNamazegVersion(
            $namozeg->id,
            $namozeg->internal_extenral,
            $tableExam->school_id
        )->id;

        /* store namozeg version for this record*/
        $tableExamNamozeg = new TableExamNamazeg();
        $tableExamNamozeg->table_exam_id = $tableExamId;
        $tableExamNamozeg->namazeg_version_id = $namazeg_version_id;
        $tableExamNamozeg->save();
    }

    private function handelFinalDiffrentExamNamazeg($tableExamId){
        $tableExam = TableExam::find($tableExamId);
        /** create or update  namozg version for this table*/
        $namozeg234 = Namazeg::where('form_number', 234)->first();
        $namazeg234_version_id = $this->storeNamazegVersion(
            $namozeg234->id,
            $namozeg234->internal_extenral,
            $tableExam->school_id
        )->id;

        $namozeg235 = Namazeg::where('form_number', 235)->first();
        $namazeg235_version_id = $this->storeNamazegVersion(
            $namozeg235->id,
            $namozeg235->internal_extenral,
            $tableExam->school_id
        )->id;

        /* store namozeg version for this record*/
        $tableExamNamozeg = new TableExamNamazeg();
        $tableExamNamozeg->table_exam_id = $tableExamId;
        $tableExamNamozeg->namazeg_version_id = $namazeg234_version_id;
        $tableExamNamozeg->save();

        $tableExamNamozeg = new TableExamNamazeg();
        $tableExamNamozeg->table_exam_id = $tableExamId;
        $tableExamNamozeg->namazeg_version_id = $namazeg235_version_id;
        $tableExamNamozeg->save();

    }

    private function handelFinalStandardExamNamazeg($tableExamId){
        /**  get school grade*/
        $schoolId= TableExam::find($tableExamId)->school_id;
        $schoolGradeId = School::find($schoolId)->school_grade_id;
        $tableExam = TableExam::find($tableExamId);
        $tableExamClassNameIds = TableExamDay::where('table_exam_id',$tableExamId)->whereNotNull('class_name_id')->pluck('class_name_id')->unique()->toArray();

        //نماذج المرحله الآبتدائية
        if($schoolGradeId == 1){
            /** create or update  namozg version for this table*/
            $namozeg238 = Namazeg::where('form_number', 238)->first();
            $namazeg238_version_id = $this->storeNamazegVersion(
                $namozeg238->id,
                $namozeg238->internal_extenral,
                $tableExam->school_id
            )->id;

            $namozeg239 = Namazeg::where('form_number', 239)->first();
            $namazeg239_version_id = $this->storeNamazegVersion(
                $namozeg239->id,
                $namozeg239->internal_extenral,
                $tableExam->school_id
            )->id;

            /* store namozeg version for this record*/
            $tableExamNamozeg = new TableExamNamazeg();
            $tableExamNamozeg->table_exam_id = $tableExamId;
            $tableExamNamozeg->namazeg_version_id = $namazeg238_version_id;
            $tableExamNamozeg->save();

            $tableExamNamozeg = new TableExamNamazeg();
            $tableExamNamozeg->table_exam_id = $tableExamId;
            $tableExamNamozeg->namazeg_version_id = $namazeg239_version_id;
            $tableExamNamozeg->save();

            //** add namozeg for each class name /////
            foreach($tableExamClassNameIds as $tableExamClassNameId){
                $namozeg248 = Namazeg::where('form_number', 248)->first();
                $namazeg248_version_id = $this->storeNamazegVersion(
                    $namozeg248->id,
                    $namozeg248->internal_extenral,
                    $tableExam->school_id
                )->id;

                $tableExamNamozeg = new TableExamNamazeg();
                $tableExamNamozeg->table_exam_id = $tableExamId;
                $tableExamNamozeg->class_name_id = $tableExamClassNameId;
                $tableExamNamozeg->namazeg_version_id = $namazeg248_version_id;
                $tableExamNamozeg->save();

            }
        }

        // نماذج المرحله الاعدادية
        if($schoolGradeId == 2){
            /** create or update  namozg version for this table*/
            $namozeg240 = Namazeg::where('form_number', 240)->first();
            $namazeg240_version_id = $this->storeNamazegVersion(
                $namozeg240->id,
                $namozeg240->internal_extenral,
                $tableExam->school_id
            )->id;

            $namozeg241 = Namazeg::where('form_number', 241)->first();
            $namazeg241_version_id = $this->storeNamazegVersion(
                $namozeg241->id,
                $namozeg241->internal_extenral,
                $tableExam->school_id
            )->id;

            /* store namozeg version for this record*/
            $tableExamNamozeg = new TableExamNamazeg();
            $tableExamNamozeg->table_exam_id = $tableExamId;
            $tableExamNamozeg->namazeg_version_id = $namazeg240_version_id;
            $tableExamNamozeg->save();

            $tableExamNamozeg = new TableExamNamazeg();
            $tableExamNamozeg->table_exam_id = $tableExamId;
            $tableExamNamozeg->namazeg_version_id = $namazeg241_version_id;
            $tableExamNamozeg->save();

            //** add namozeg for each class name /////
            foreach($tableExamClassNameIds as $tableExamClassNameId){
                $namozeg248 = Namazeg::where('form_number', 248)->first();
                $namazeg248_version_id = $this->storeNamazegVersion(
                    $namozeg248->id,
                    $namozeg248->internal_extenral,
                    $tableExam->school_id
                )->id;

                $tableExamNamozeg = new TableExamNamazeg();
                $tableExamNamozeg->table_exam_id = $tableExamId;
                $tableExamNamozeg->class_name_id = $tableExamClassNameId;
                $tableExamNamozeg->namazeg_version_id = $namazeg248_version_id;
                $tableExamNamozeg->save();

            }

        }

        // نماذج المرحله الثانوية
        if($schoolGradeId == 3 && count($tableExamClassNameIds) > 1){

            $checkClassCategory= TableExamDay::where('table_exam_id',$tableExamId)->pluck('class_name_category')->unique()->toArray();
            if($checkClassCategory){
                /// نظام المسارات

                $namozeg246 = Namazeg::where('form_number', 246)->first();
                $namazeg246_version_id = $this->storeNamazegVersion(
                    $namozeg246->id,
                    $namozeg246->internal_extenral,
                    $tableExam->school_id
                )->id;

                $namozeg247 = Namazeg::where('form_number', 247)->first();
                $namazeg247_version_id = $this->storeNamazegVersion(
                    $namozeg247->id,
                    $namozeg247->internal_extenral,
                    $tableExam->school_id
                )->id;

                /* store namozeg version for this record*/
                $tableExamNamozeg = new TableExamNamazeg();
                $tableExamNamozeg->table_exam_id = $tableExamId;
                $tableExamNamozeg->namazeg_version_id = $namazeg246_version_id;
                $tableExamNamozeg->save();

                $tableExamNamozeg = new TableExamNamazeg();
                $tableExamNamozeg->table_exam_id = $tableExamId;
                $tableExamNamozeg->namazeg_version_id = $namazeg247_version_id;
                $tableExamNamozeg->save();

            }else{
                // النظام العام
                /** create or update  namozg version for this table*/
                $namozeg242 = Namazeg::where('form_number', 242)->first();
                $namazeg242_version_id = $this->storeNamazegVersion(
                    $namozeg242->id,
                    $namozeg242->internal_extenral,
                    $tableExam->school_id
                )->id;

                $namozeg243 = Namazeg::where('form_number', 243)->first();
                $namazeg243_version_id = $this->storeNamazegVersion(
                    $namozeg243->id,
                    $namozeg243->internal_extenral,
                    $tableExam->school_id
                )->id;

                /* store namozeg version for this record*/
                $tableExamNamozeg = new TableExamNamazeg();
                $tableExamNamozeg->table_exam_id = $tableExamId;
                $tableExamNamozeg->namazeg_version_id = $namazeg242_version_id;
                $tableExamNamozeg->save();

                $tableExamNamozeg = new TableExamNamazeg();
                $tableExamNamozeg->table_exam_id = $tableExamId;
                $tableExamNamozeg->namazeg_version_id = $namazeg243_version_id;
                $tableExamNamozeg->save();

                //** add namozeg for each class name /////
                foreach($tableExamClassNameIds as $tableExamClassNameId){
                    $namozeg248 = Namazeg::where('form_number', 248)->first();
                    $namazeg248_version_id = $this->storeNamazegVersion(
                        $namozeg248->id,
                        $namozeg248->internal_extenral,
                        $tableExam->school_id
                    )->id;

                    $tableExamNamozeg = new TableExamNamazeg();
                    $tableExamNamozeg->table_exam_id = $tableExamId;
                    $tableExamNamozeg->class_name_id = $tableExamClassNameId;
                    $tableExamNamozeg->namazeg_version_id = $namazeg248_version_id;
                    $tableExamNamozeg->save();

                }
            }
        }

        if($schoolGradeId == 3 && count($tableExamClassNameIds)  == 1){
            /// الصف الاول الثانوي
            $checkClassName = ClassName::where('id', reset($tableExamClassNameIds))->first();

            if($checkClassName->name == 'الأول الثانوي'){
                $namozeg244 = Namazeg::where('form_number', 244)->first();
                $namazeg244_version_id = $this->storeNamazegVersion(
                    $namozeg244->id,
                    $namozeg244->internal_extenral,
                    $tableExam->school_id
                )->id;

                $namozeg245 = Namazeg::where('form_number', 245)->first();
                $namazeg245_version_id = $this->storeNamazegVersion(
                    $namozeg245->id,
                    $namozeg245->internal_extenral,
                    $tableExam->school_id
                )->id;

                /* store namozeg version for this record*/
                $tableExamNamozeg = new TableExamNamazeg();
                $tableExamNamozeg->table_exam_id = $tableExamId;
                $tableExamNamozeg->namazeg_version_id = $namazeg244_version_id;
                $tableExamNamozeg->save();

                $tableExamNamozeg = new TableExamNamazeg();
                $tableExamNamozeg->table_exam_id = $tableExamId;
                $tableExamNamozeg->namazeg_version_id = $namazeg245_version_id;
                $tableExamNamozeg->save();
            }

        }

    }

    private function smsMonthlyExames($school_id,$class_id,$start_date){
        $start_day =   Carbon::createFromFormat('d/m/Y', $start_date)->translatedFormat('D', 'ar');

        $message_text = "تم اصدار جداول الاختبارات  الفترية , ستبدأ الاختبارات من يوم  ".$start_date.' , '.$start_day.' والله الموفق والمستعان .';
        $students = Student::where('school_id', $school_id)->where('class_id', $class_id)->get();
        $school = School::findOrFail($school_id);
        foreach ($students as $student) {
            static::sendSMSNotification($student?->phone, $message_text, $school->user_id);
        }
    }

    private function smsStandredFinalExames($school_id,$class_name_id,$start_date){
        $message_text = "تم اصدار جداول الاختبارات النهائية,نرجو منكم متابعة استعدادات أبنائكم لضمان جاهزيتهم للاختبارات , والله الموفق. موعد البداية: $start_date";
        $classIds = SchoolClass::where('school_id',$school_id)->where('class_name_id',$class_name_id)->pluck('id')->toArray();
        $students = Student::where('school_id', $school_id)->whereIn('class_id', $classIds)->get();
        $school = School::findOrFail($school_id);
        foreach ($students as $student) {
            static::sendSMSNotification($student?->phone, $message_text, $school->user_id);
        }
    }

    private function smsDifferentFinalExames($school_id,$class_id,$start_date){
        $message_text = "تم اصدار جداول الاختبارات النهائية,نرجو منكم متابعة استعدادات أبنائكم لضمان جاهزيتهم للاختبارات , والله الموفق. موعد البداية: $start_date";
        $students = Student::where('school_id', $school_id)->where('class_id', $class_id)->get();
        $school = School::findOrFail($school_id);
        foreach ($students as $student) {
            static::sendSMSNotification($student?->phone, $message_text, $school->user_id);
        }
    }

}
