<?php

namespace App\Http\Controllers\Api\ManagerDashboard\StudentAbsenceAndLateness;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ManagerDashboard\AbsenseAndLateness\StudentAttendanceResource;
use App\Traits\SmsTrait;
use Illuminate\Http\Request;
use App\Models\AdminDashboard\School\SchoolClass;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\AbsenseAndLateness\StudentAttendance;
use Illuminate\Support\Facades\DB;
use Tymon\JWTAuth\Facades\JWTAuth;

class GeneralStudentAbsenceController extends Controller
{
    use SmsTrait;
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:school-student-attendance']);
    }

    public function getDayRegisteredClasses(Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $RegisteredClasses = StudentAttendance::select('class_names.name', 'school_classes.class_name_id', 'school_classes.class_number')
            ->join('school_classes', 'school_classes.id', 'students_attendance.class_id')
            ->join('class_names', 'class_names.id', 'school_classes.class_name_id')
            ->where('students_attendance.school_id', $request->school_id)
            ->where('students_attendance.semester_id', $semester->id)
            ->where('students_attendance.date', $date)
            ->whereNotNull('students_attendance.status')
            ->groupBy('students_attendance.class_id')
            ->get();

        $RegisteredClassesCount = count($RegisteredClasses);
        $AllClasses = SchoolClass::where('school_id', $request->school_id)->count();

        return response()->json([
            'RegisteredClasses' => $RegisteredClasses,
            'RegisteredClassesCount' => $RegisteredClassesCount,
            'AllClassesCount' => $AllClasses,
            'message' => trans(
                'api.list of Registered classes'
            )
        ]);
    }

    public function getDayStudents(Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $studentIds = Student::where('school_id', $request->school_id);

        if ($request->student_name) {
            $studentIds->where('name', 'LIKE', "{$request->student_name}%");
        }

        if ($request->class_name_id && !$request->class_number) {
            $class_ids = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $request->class_name_id)->pluck('id')->toArray();
            $studentIds->whereIn('class_id', $class_ids);
        }

        if ($request->class_number) {
            $class = SchoolClass::where('school_id', $request->school_id)->where('class_name_id', $request->class_name_id)->where('class_number', $request->class_number)->firstOrFail();
            $studentIds->where('class_id', $class->id);
        }

        $studentIds = $studentIds->pluck('id')->toArray();

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $checkIfCreated = StudentAttendance::where('date', $date)->where('school_id', $request->school_id)->where('semester_id', $semester->id)->exists();
        if (!$checkIfCreated) {
            try {
                DB::beginTransaction();
                foreach ($studentIds as $student_id) {
                    $studentAttendance = new StudentAttendance();
                    $studentAttendance->school_id = $request->school_id;
                    $studentAttendance->student_id = $student_id;
                    $studentAttendance->semester_id = $semester->id;
                    $studentAttendance->class_id = Student::findOrFail($student_id)?->class_id;
                    $studentAttendance->status = 'attended';
                    $studentAttendance->date = $date;
                    $studentAttendance->date_type = 'coptic';
                    $studentAttendance->save();
                }
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }
        }

        // Paginate results with a default value if not provided
        $perPage = $request->per_page ?? 15;

        $studentAttendance = StudentAttendance::whereIn('student_id', $studentIds)->where('school_id', $request->school_id)->where('semester_id', $semester->id)->where('date', 'LIKE', $date)->paginate($perPage);
        $studentAttendances = StudentAttendanceResource::collection($studentAttendance);

        // All Top Statistics
        $schoolStudentsCount = count($studentIds);
        $attendedSchoolStudents = StudentAttendance::where('school_id', $request->school_id)->where('semester_id', $semester->id)->where('status', 'attended')->where('date', $date)->count();
        $absentSchoolStudents = StudentAttendance::where('school_id', $request->school_id)
            ->where('date', $date)
            ->where('semester_id', $semester->id)
            ->where(function ($q) {
                $q->where('status', 'absent')
                    ->orWhere('status', 'absent_with_excuse');
            })->count();
        $StudentsNotRegisteredYet = $schoolStudentsCount - ($attendedSchoolStudents + $absentSchoolStudents);
        $StudentsNotRegisteredYet = $StudentsNotRegisteredYet > 0 ? $StudentsNotRegisteredYet : 0;
        $RegisteredClasses = StudentAttendance::select('class_id')->where('school_id', $request->school_id)
            ->where('semester_id', $semester->id)
            ->where('date', $date)
            ->whereNotNull('status')
            ->groupBy('class_id')
            ->get();
        $AllClasses = SchoolClass::where('school_id', $request->school_id)->count();

        return response()->json([
            'studentAttendances' => $studentAttendances,
            'schoolStudentsCount' => $schoolStudentsCount,
            'attendedSchoolStudentsCount' => $attendedSchoolStudents,
            'absentSchoolStudentsCount' => $absentSchoolStudents,
            'StudentsNotRegisteredYetCount' => $StudentsNotRegisteredYet,
            'RegisteredClassesCount' => count($RegisteredClasses),
            'AllClassesCount' => $AllClasses,
            'message' => trans(
                'api.list of students attendance'
            )
        ]);
    }

    /** autocomplete the school Student name */
    public function studentAutocompleteName(Request $request)
    {
        $results = Student::where('school_id', $request->id)->where('name', 'LIKE', "{$request->student_name}%")
            ->select('id', 'name')
            ->get();
        return response()->json($results);
    }

    /** change status of attendance to particular employee */
    public function changeAttendanceStatus(Request $request)
    {
        $rules = [
            'school_id' => 'required|exists:schools,id',
            'date_type' => 'nullable|in:coptic,hijri',
            'records' => 'required|array|min:1',
            'records.*.student_id' => 'required|exists:school_students,id',
            'records.*.status' => 'required|in:attended,absent,absent_with_excuse',
        ];

        $request->validate($rules);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        foreach ($request->input('records') as $record) {
            $studentAttendance = StudentAttendance::where('school_id', $request->school_id)
                ->where('semester_id', $semester->id)
                ->where('student_id', $record['student_id'])
                ->where('date', $date)
                ->firstOrFail();

            if ($record['status'] != "attended") {
                $user =  JWTAuth::parseToken()->authenticate();
                $message_text = "تم تسجيل غياب ل " . $studentAttendance->student . " هذا اليوم, يرجى متابعة المواظبة.";
                static::sendSMSNotification($studentAttendance->student->phone, $message_text, $user->manager_id);
            }
            $studentAttendance->update([
                'status' => $record['status'],
                'absence_reason' => $record['absence_reason'] ?? null,
            ]);
        }

        return response()->json(['message' => trans('api.student attendance updated successfully')]);
    }

    /** تغييب الكل */
    public function makeAllStudentsAbsent(Request $request)
    {
        $request->validate([
            'school_id' => 'required|exists:schools,id',
            // 'date' => 'required',
            'date_type' => 'nullable|in:coptic,hijri',
        ]);

        if ($request->date) {
            if ($request->date_type == 'hijri') {
                $date = Helper::getCoptic($request->date);
            } else {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date)->format('d/m/Y');
            }
        } else {
            $date = date('d/m/Y');
        }

        $semester = Helper::getCurrentSchoolSemester($date);
        if (!$semester) {
            return response()->json(['error' => trans('api.there is no semester registered for this date')], 404);
        }

        $studentAttendances = StudentAttendance::where('school_id', $request->school_id)->where('semester_id', $semester->id)->where('date', $date)->get();
        foreach ($studentAttendances as $studentAttendance) {
            $studentAttendance->update([
                'status' => 'absent',
                'absence_reason' => null,
            ]);
        }
        return response()->json(['message' => trans('api.all student Attendance updated to Absent Successfully')]);
    }
}
