<?php

namespace App\Http\Controllers\Api\AdminDashboard\Report;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\School\SchoolShortInfoResource;
use App\Models\AdminDashboard\AreaAndCities\Area;
use App\Models\AdminDashboard\AreaAndCities\City;
use App\Models\AdminDashboard\School\School;
use App\Models\AdminDashboard\School\SchoolEmployee;
use App\Models\AdminDashboard\School\Student;
use App\Models\ManagerDashboard\Settings\Subscription;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\Activitylog\Models\Activity;

class ReportController extends Controller
{
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        //$this->middleware(['auth:api','permission:admin-dashboard-report']);
    }


    public function getFinancialStatistics(Request $request){
        $total_subscriptions = Subscription::count();
        $paid_subscriptions = Subscription::where('paid_status','confirm')->count();
        $refused_subscriptions = Subscription::where('paid_status','refuse')->count();
        $bank_transfers = Subscription::where('paid_type','bank_transfer')->count();
        $pending_bank_transfers = Subscription::where('paid_status','panding')->where('paid_type','bank_transfer')->count();
        $paid_with_bank_transfers = Subscription::where('paid_status','confirm')->where('paid_type','bank_transfer')->count();
        $paid_with_payment_getway = Subscription::where('paid_type','payment_gateway')->count();
        $discount = Subscription::sum('coupon_discount');
        $profits = 0;
        $taxes = Subscription::sum('tax_fees');

        $data= [
            'total_subscriptions' => $total_subscriptions,
            'paid_subscriptions' => $paid_subscriptions,
            'refused_subscriptions' => $refused_subscriptions,
            'bank_transfers' => $bank_transfers,
            'pending_bank_transfers' => $pending_bank_transfers,
            'paid_with_bank_transfers' => $paid_with_bank_transfers,
            'paid_with_payment_getway' => $paid_with_payment_getway,
            'discount' => $discount,
            'profits' => $profits,
            'taxes' => $taxes,
        ];

        return response()->json(['data' => $data, 'message' => trans('api.financial statistics')]);


    }

    public function getSchoolsStatistics(Request $request){
        $schools = School::query();

        $not_shared_school = School::select('user_id')->groupBy('user_id')->havingRaw('COUNT(*) = 1');
        $shared_school = School::select('user_id')->groupBy('user_id')->havingRaw('COUNT(*) = 2');

        $governmentalSchools = School::where('education_type','governmental');
        $familySchools = School::where('education_type','family');

        $total_students = Student::query();
        $total_school_mangers = SchoolEmployee::where('type','adminstrative')->where('current_work_id',1);
        $total_teachers = SchoolEmployee::where('type','teacher')->where('current_work_id',2);


        $schools_count =$schools->count();
        $not_shared_school_count = $not_shared_school->count();
        $shared_school_count = $shared_school->count();
        $governmentalSchools_count = $governmentalSchools->count();
        $familySchools_count = $familySchools->count();

        $total_students_count = $total_students->count();
        $total_school_mangers_count = $total_school_mangers->count();
        $total_teachers_count = $total_teachers->count();



        $areas_count = Area::count();
        $cities_count = City::count();

        /** schools custom filter*/
        if ($request->area_ids) {
            $schools = $schools->whereIn('area_id', explode(',',$request->area_ids));
            $not_shared_school_count = $not_shared_school->whereIn('area_id', explode(',',$request->area_ids))->count();
            $shared_school_count = $shared_school->whereIn('area_id', explode(',',$request->area_ids))->count();
            $governmentalSchools_count = $governmentalSchools->whereIn('area_id', explode(',',$request->area_ids))->count();
            $familySchools_count = $familySchools->whereIn('area_id', explode(',',$request->area_ids))->count();

            $schoolIds = $schools->whereIn('area_id', explode(',',$request->area_ids))->pluck('id')->toArray();

            $total_students_count = $total_students->whereIn('school_id',$schoolIds)->count();
            $total_school_mangers_count = $total_school_mangers->whereIn('school_id',$schoolIds)->count();
            $total_teachers_count = $total_teachers->whereIn('school_id',$schoolIds)->count();
        }

        if ($request->city_ids) {
            $schools = $schools->whereIn('city_id', explode(',',$request->city_ids));
            $not_shared_school_count = $not_shared_school->whereIn('city_id', explode(',',$request->city_ids))->count();
            $shared_school_count = $shared_school->whereIn('city_id', explode(',',$request->city_ids))->count();
            $governmentalSchools_count = $governmentalSchools->whereIn('city_id', explode(',',$request->city_ids))->count();
            $familySchools_count = $familySchools->whereIn('city_id', explode(',',$request->city_ids))->count();

            $schoolIds = $schools->whereIn('city_id', explode(',',$request->city_ids))->pluck('id')->toArray();

            $total_students_count = $total_students->whereIn('school_id',$schoolIds)->count();
            $total_school_mangers_count = $total_school_mangers->whereIn('school_id',$schoolIds)->count();
            $total_teachers_count = $total_teachers->whereIn('school_id',$schoolIds)->count();

        }

        if ($request->gender) {
            $schools = $schools->where('gender', $request->gender);
            $not_shared_school_count = $not_shared_school->where('gender', $request->gender)->count();
            $shared_school_count = $shared_school->where('gender', $request->gender)->count();
            $governmentalSchools_count = $governmentalSchools->where('gender', $request->gender)->count();
            $familySchools_count = $familySchools->where('gender', $request->gender)->count();

            $schoolIds = $schools->where('gender', $request->gender)->pluck('id')->toArray();

            $total_students_count = $total_students->whereIn('school_id',$schoolIds)->count();
            $total_school_mangers_count = $total_school_mangers->whereIn('school_id',$schoolIds)->count();
            $total_teachers_count = $total_teachers->whereIn('school_id',$schoolIds)->count();
        }


        if ($request->school_grade_id) {
            $schools->where('school_grade_id',$request->school_grade_id);
        }

        if ($request->education_type) {
            $schools->where('education_type', $request->education_type);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $filterData = SchoolShortInfoResource::collection($schools->paginate($perPage));

        $data= [
            'not_shared_school_count' => $not_shared_school_count,
            'shared_school_count' => $shared_school_count,
            'governmental_school_count' => $governmentalSchools_count,
            'family_school_count' => $familySchools_count,
            'total_students' => $total_students_count,
            'total_school_mangers' => $total_school_mangers_count,
            'total_teachers' => $total_teachers_count,
            'areas_count' => $areas_count,
            'cities_count' => $cities_count,
            'schools'=>$filterData,
        ];

        return response()->json(['data' => $data, 'message' => trans('api.schools statistics')]);

    }

    public function filterSchools(Request $request){

        // Start with the base query
        $data = School::query();
        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            return !in_array($key, ['per_page', 'school_grade','gender', 'education_type','area_ids' , 'city_ids']);
        }, ARRAY_FILTER_USE_KEY);
        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\School', $filterData);
        }
        $data->get();

        // custom filter////
        if ($request->school_grade_id) {
            $data->where('school_grade_id',$request->school_grade_id);
        }

        if ($request->education_type) {
            $data->where('education_type', $request->education_type);
        }


        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;
        $filterData = SchoolShortInfoResource::collection($data->paginate($perPage));

        return response()->json(['schools' => $filterData, 'message' => trans('api.all school records')]);
    }

    public function getClientsStatistics(Request $request)
    {
        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        if(isset($start_date)) {
            if ($request['start_date_type'] == "hijri") {
                $start_date = Helper::getCoptic($start_date);
            } elseif($request['start_date_type'] == "coptic") {
                $start_date = Carbon::createFromFormat('d/m/Y',$start_date);
            }
        }

        if(isset($end_date)) {
            if ($request['end_date_type'] == "hijri") {
                $end_date = Helper::getCoptic($end_date);
            } elseif($request['end_date_type'] == "coptic") {
                $end_date = Carbon::createFromFormat('d/m/Y',$end_date);
            }
        }

        if(isset($start_date) && isset($end_date)) {
            $allClientsCount = User::where('user_type','school_manager')->whereBetween('created_at',[$start_date,$end_date])->count();
            $subscripedClientsCount = Subscription::join('users', 'users.id', 'subscriptions.user_id')
                ->select('user_id')
                ->where('subscriptions.status', 'Subscribed')
                ->where('service_type', 'system')
                ->where('users.status', 'active')
                ->whereBetween('subscriptions.created_at',[$start_date,$end_date])
                ->count();

            $unsubscripedClientsCount = Subscription::join('users', 'users.id', 'subscriptions.user_id')
                ->select('user_id')
                ->where('subscriptions.status', 'not_subscribed')
                ->where('service_type', 'system')
                ->where('users.status', 'active')
                ->whereBetween('subscriptions.created_at',[$start_date,$end_date])
                ->count();

            $experiemenstalClientsCount = Subscription::join('users', 'users.id', 'subscriptions.user_id')
                ->select('user_id')
                ->where('subscriptions.status', 'Experimental')
                ->where('service_type', 'system')
                ->where('users.status', 'active')
                ->whereBetween('subscriptions.created_at',[$start_date,$end_date])
                ->count();

            $endSoonClientsCount = Subscription::join('users', 'users.id', 'subscriptions.user_id')
                ->select('user_id')
                ->whereRaw("STR_TO_DATE(end_date, '%d/%m/%Y') >= ?", [Carbon::now()])
                ->whereRaw("DATEDIFF(STR_TO_DATE(end_date, '%d/%m/%Y'), ?) <= 30", [Carbon::now()])
                ->where('service_type', 'system')
                ->where('users.status', 'active')
                ->whereBetween('subscriptions.created_at',[$start_date,$end_date])
                ->count();

            $visits = User::whereBetween('created_at',[$start_date,$end_date])->count();

            $actions = Activity::where('event', 'LIKE' ,"%store%")
                ->orWhere('event', 'LIKE' ,"%update%")
                ->orWhere('event', 'LIKE' ,"%archive%")
                ->orWhere('event', 'LIKE' ,"%delete%")
                ->whereBetween('created_at',[$start_date,$end_date])
                ->count();

            $usersCount = User::whereBetween('created_at',[$start_date,$end_date])->count();

        } else {

            $allClientsCount = User::where('user_type','school_manager')->count();
            $subscripedClientsCount = Subscription::join('users', 'users.id', 'subscriptions.user_id')
                ->select('user_id')
                ->where('subscriptions.status', 'Subscribed')
                ->where('users.status', 'active')
                ->groupBy('user_id')
                ->count();

            $unsubscripedClientsCount = Subscription::join('users', 'users.id', 'subscriptions.user_id')
                ->select('user_id')
                ->where('subscriptions.status', 'not_subscribed')
                ->where('service_type', 'system')
                ->where('users.status', 'active')
                ->count();

            $experiemenstalClientsCount = Subscription::join('users', 'users.id', 'subscriptions.user_id')
                ->select('user_id')
                ->where('subscriptions.status', 'Experimental')
                ->where('service_type', 'system')
                ->where('users.status', 'active')
                ->count();

            $endSoonClientsCount = Subscription::join('users', 'users.id', 'subscriptions.user_id')
                ->select('user_id')
                ->whereRaw("STR_TO_DATE(end_date, '%d/%m/%Y') >= ?", [Carbon::now()])
                ->whereRaw("DATEDIFF(STR_TO_DATE(end_date, '%d/%m/%Y'), ?) <= 30", [Carbon::now()])
                ->where('service_type', 'system')
                ->where('users.status', 'active')
                ->count();

            $visits = User::count();

            $actions = Activity::where('event', 'LIKE' ,"%store%")
                ->orWhere('event', 'LIKE' ,"%update%")
                ->orWhere('event', 'LIKE' ,"%archive%")
                ->orWhere('event', 'LIKE' ,"%delete%")
                ->count();

            $usersCount = User::count();
        }

        $data = [
            'allClientsCount' => $allClientsCount,
            'subscripedClientsCount' => $subscripedClientsCount,
            'unsubscripedClientsCount' => $unsubscripedClientsCount,
            'experiemenstalClientsCount' => $experiemenstalClientsCount,
            'endSoonClientsCount' => $endSoonClientsCount,
            'visits' => $visits,
            'actions' => $actions,
            'usersCount' => $usersCount,
        ];

        return response()->json(['data' => $data, 'message' => trans('api.clients statistics')]);
    }

    public function getUsersMonthlyCountPerYear($year = null)
    {
        if($year != null) {
            $year = Carbon::now()->format('Y');
        }

        $results = User::select(DB::raw('MONTH(created_at) as month'), DB::raw('COUNT(*) as count'))
            ->whereYear('created_at', $year)
            ->groupBy('month')
            ->orderBy('month', 'asc')
            ->get();

        // Transform the results to an associative array with month names as keys
        $monthlyCounts = [];
        foreach ($results as $result) {
            $monthName = Carbon::createFromDate($year, $result->month, 1)->format('F');
            $monthlyCounts[$monthName] = $result->count;
        }

        return response()->json(['usersMonthlyCounts' => $monthlyCounts, 'message' => trans('api.users count for each month of specific year')]);
    }

}

