<?php

namespace App\Http\Controllers\Api\AdminDashboard\Coupons;

use App\Exports\CouponsExport;
use App\Exports\CouponUsageHistoryExport;
use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\AdminDashboard\Coupons\StoreCouponRequest;
use App\Http\Requests\AdminDashboard\Coupons\UpdateCouponRequest;
use App\Http\Requests\UpdateStatusRequest;
use App\Http\Resources\AdminDashboard\Coupons\CouponResource;
use App\Http\Resources\AdminDashboard\Coupons\CouponUsageHistoryResource;
use App\Models\AdminDashboard\Coupons\Coupon;
use App\Models\ManagerDashboard\Settings\Subscription;
use App\Traits\LogsActivity;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;

class CouponController extends Controller
{
    use LogsActivity;

    /**
     * Display a listing of the resource.
     */

    public function getAll(Request $request)
    {
        // Start with the base query
        $data = Coupon::with('products');

        // Retrieve all query parameters
        $queryParams = $request->query();

        // Filter data is directly from query parameters, excluding `per_page`
        $filterData = array_filter($queryParams, function ($key) {
            return $key !== 'per_page';
        }, ARRAY_FILTER_USE_KEY);

        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\AdminDashboard\Coupons\Coupon', $filterData);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        $filterData = CouponResource::collection($data->paginate($perPage));


        return response()->json([
            'coupons' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCouponRequest $request)
    {


        $validatedData = $request->validated();

        $productIds = $validatedData['product_ids'];

        unset($validatedData['product_ids']);

        $coupon = Coupon::create($validatedData);

        $coupon->products()->attach($productIds);
        $this->AdminlogActivity('Coupon', 'store', '/coupons/' . $coupon->id . '/details/');

        return response()->json(['message' => trans('api.record stored successflly')], 200);
    }

    /**
     * Display the specified resource.
     */
    public function getById(Coupon $coupon)
    {

        $coupon = new CouponResource($coupon);

        return response()->json(['coupon' => $coupon], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateCouponRequest $request, Coupon $coupon)
    {


        $validatedData = $request->validated();
        $productIds = $validatedData['product_ids'];

        unset($validatedData['product_ids']);
        $coupon->update($validatedData);
        $coupon->products()->sync($productIds);
        $this->AdminlogActivity('Coupon', 'update', '/coupons/' . $coupon->id . '/details/');

        return response()->json(['message' => trans('api.record updated successflly')], 200);
    }

    public function changeStatus(Request $request, Coupon $coupon)
    {
        $validatedData = $request->validate([
            'coupon_status' => 'required|in:active,expired',
        ]);

        $coupon->coupon_status = $validatedData['coupon_status'];
        $coupon->save();
        return response()->json(['message' => trans('api.record updated successflly')], 200);
    }

    /**
     * Soft Delete.
     */
    public function archive(Coupon $coupon)
    {
        $coupon->delete();
        $this->AdminlogActivity('Coupon', 'archive');

        return response()->json(['message' => trans('api.record archived successflly')], 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Coupon $coupon)
    {

        $coupon->forceDelete();
        $this->AdminlogActivity('Coupon', 'delete');


        return response()->json(['message' => trans('api.record deleted successflly')], 200);
    }

    /**
     * restore the specified resource which soft Deleted.
     */
    public function restore(Coupon $coupon)
    {

        $coupon->restore();
        $this->AdminlogActivity('Coupon', 'restore');
        return response()->json(['message' => trans('api.record restored successflly')], 200);
    }


    public function export(Request $request)
    {
        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        if (isset($request['date_type']) && isset($request['start_date']) && isset($request['end_date'])) {
            $start_date = $request['start_date'];
            $end_date = $request['end_date'];

            if ($request['date_type'] == "hijri") {
                $start_date = Helper::getCoptic($request['start_date']);
                $end_date = Helper::getCoptic($request['end_date']);
            }
        }

        return Excel::download(new CouponsExport($start_date, $end_date), 'Coupons.xlsx');
    }

    public function getCouponUsageHistory(Coupon $coupon)
    {
        $perPage = $queryParams['per_page'] ?? 15;

        $subscriptions = Subscription::where('coupon_id', $coupon->id)->paginate($perPage);

        $response = CouponUsageHistoryResource::collection($subscriptions);


        return response()->json(['data' => $response]);
    }

    public function exportCouponUsageHistory(Request $request, Coupon $coupon)
    {
        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');
        if (isset($request['start_date_type']) && isset($request['start_date'])) {
            $start_date = $request['start_date'];

            if ($request['start_date_type'] == "hijri") {
                $start_date = Helper::getCoptic($request['start_date']);
            }

        }

        if (isset($request['end_date_type']) && isset($request['end_date'])) {
            $end_date = $request['end_date'];

            if ($request['end_date_type'] == "hijri") {
                $end_date = Helper::getCoptic($request['end_date']);
            }
        }


        return Excel::download(new CouponUsageHistoryExport($start_date, $end_date, $coupon), 'CouponUsageHistory.xlsx');
    }
}
