<?php

namespace App\Http\Controllers\Api\AdminDashboard\Clients;

use App\Exports\ClientExport;
use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Resources\User\UserResource;
use App\Models\AdminDashboard\School\School;
use App\Traits\SmsTrait;
use Illuminate\Http\Request;
use App\Http\Resources\AdminDashboard\Clients\ClientResource;
use App\Models\ManagerDashboard\Settings\Subscription;
use App\Models\User;
use App\Traits\LogsActivity;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;
use JWTAuth;
use Illuminate\Support\Facades\Validator;

class ClientController extends Controller
{

    use LogsActivity;
    use SmsTrait;
    /* to uses the class methods
        -you should be authenticated
        -you should be authorized and have this permission
    */
    public function __construct()
    {
        // $this->middleware(['auth:api', 'jwt.verify','permission:admin-clients']);
    }

    /*
        -this method will retur n all Model recoreds
    */
    public function getAll(Request $request)
    {
        $data = Subscription::where('service_type','system')->select(
            'id',
            'user_id',
            'status',
            'end_date',
            'cancellation_request_status'
        );

        $queryParams = $request->query();
        $perPage = $queryParams['per_page'] ?? 15;

        if (isset($request->client_name)) {
            $data->join('users', 'users.id', '=', 'subscriptions.user_id')
                ->where('users.name', 'like', "%{$request->client_name}%");
        }

        if (isset($request->client_type)) {
            $data->where('subscriptions.status', $request->client_type);
        }

        if (isset($request->account_status)) {
            $data->join('users', 'users.id', '=', 'subscriptions.user_id')
                ->where('users.status', $request->account_status);
        }

        $filterData = ClientResource::collection($data->paginate($perPage));

        return response()->json([
            'clients' => $filterData,
            'message' => trans('api.list of all available clients'),
        ]);
    }

    public function exportClients(Request $request)
    {
        $start_date = $request->input('start_date');
        $end_date = $request->input('end_date');

        if (isset($request['start_date_type']) && $request['start_date_type'] == "hijri" && isset($start_date)) {
            $start_date = Helper::getCoptic($request['start_date']);
        }

        if (isset($request['end_date_type']) && $request['end_date_type'] == "hijri" && isset($end_date)) {
            $end_date = Helper::getCoptic($request['end_date']);
        }

        return Excel::download(new ClientExport($start_date, $end_date), 'clients.xlsx', \Maatwebsite\Excel\Excel::XLSX, [
            'Content-Disposition' => 'attachment; filename="clients_' . time() . '.xlsx"',
        ]);
    }

    public function changeStatus(User $user)
    {
        if ($user->status == 'active') {
            $user->update(['status' => 'inactive']);
            $message = 'disabled';
        } else {
            $user->update(['status' => 'active']);
            $message = 'activated';
        }

        return response()->json([
            'message' => trans("api.account $message successfully"),
        ]);
    }

    public function userSubscriptions(User $user,Request $request)
    {
        $perPage = $request->per_page ?? 15;
        $data = Subscription::with('user')->select('id','user_id','start_date','end_date')
            ->where('service_type','system')
            ->whereIn('status',['subscribed','Experimental','not_subscribed'])
            ->where('user_id', $user->id)
            ->paginate($perPage);

        $subscriptions = [];
        foreach ($data as $record) {

            if (Carbon::createFromFormat('d/m/Y', $record->end_date) > Carbon::now()) {
                $remainingDays = Carbon::createFromFormat('d/m/Y', $record->end_date)->diffInDays(Carbon::now());
            } else {
                $remainingDays = 0;
            }

            $subscriptions[] = [
                'start_date' => $record->start_date,
                'end_date' => $record->end_date,
                'remaining_days' => $remainingDays,
                'account_status' => $record->user->status,
            ];

        }

        $subscriptions = $data->setCollection(collect($subscriptions));

        return response()->json([
            'subscriptions' => $subscriptions,
            'message' => trans('api.list of all available subscriptions'),
        ]);
    }

    public function cancelSubscription(Request $request)
    {
        Subscription::where('user_id', $request->user_id)->where('service_type','system')->first()->update(['status' => 'not_subscribed']);

        User::find($request->user_id)->update(['status' => 'canceled']);

        return response()->json([
            'message' => trans("api.subscription canceled successfully"),
        ]);
    }

    public function restoreSubscription(Request $request)
    {
        Subscription::where('user_id', $request->user_id)->where('service_type','system')->first()->update(['status' => 'Subscribed']);;

        User::find($request->user_id)->update(['status' => 'active']);

        return response()->json([
            'message' => trans("api.subscription restored successfully"),
        ]);
    }


    public function sendSystemAccountCredentials(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }
        $school = School::where('user_id',$user->id)->where('id',$request->school_id)->firstOrFail();

        $user = User::find($request->user_id);
        if (!$user) {
            return response()->json(['message' => trans('api.User not found.')], 404);
        }

        //send sms with system account credentials//
        $this->sendSystemAccount($user->phone,$user->id);

        return response()->json(['message'=>trans('api.account sent successfully')]);
    }

    public function updateProfile(Request $request){
        $user = User::findOrfail($request->user_id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'phone' => 'required|string|regex:/^05\d{8}$/|unique:users,phone,' . $user->id,
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors();
            foreach ($errors->all() as $message) {
                $error[] =  trans("api.$message");
            }
            return response()->json(['message' => trans('api.error entering data'), 'errors' => $validator->errors()], 400);
        }

        if($user->phone != $request->phone){
            $auth_user =  JWTAuth::parseToken()->authenticate();
            $message_text = " تم تغيير رقم الجوال بنجاح.";
            static::sendSMSNotification($request->phone, $message_text, $auth_user->manager_id);
        }

        $user->name=$request->name;
        $user->email=$request->email;
        $user->phone=$request->phone;
        $user->save();

        // $user = new UserResource ($user);

        return response()->json(['message'=>trans('api.user data updated successfully')]);
    }

    public function acceptCancellation(Subscription $subscription)
    {
        $subscription->update([
            'cancellation_request_status' => 'accepted',
            'admin_response' => 'accepted',
            'status' => 'not_subscrped',
        ]);

        $user = User::where('id',$subscription->user_id)->firstOrFail();
        $user->update(['status' => 'canceled_by_user']);

        return response()->json(['message'=>'تم الموافقه علي طلب الغاء الاشتراك بنجاح']);
    }

    public function refuseCancellation(Subscription $subscription)
    {
        $subscription->update([
            'cancellation_request_status' => 'refused',
            'admin_response' => 'refused',
        ]);

        return response()->json(['message'=>'تم رفض  طلب الغاء الاشتراك ']);

    }

}
