<?php

namespace App\Http\Requests\School;

use App\Helpers\Helper;
use App\Models\AdminDashboard\School\SchoolSemester;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class SchoolSemesterVacationStoreRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $date = $this->date; // The date is already formatted in prepareForValidation()
        
        // Check if the date falls within any valid semester
        $semester = SchoolSemester::whereRaw('STR_TO_DATE(start_date, "%d/%m/%Y") <= ?', [$date])
        ->whereRaw('STR_TO_DATE(end_date, "%d/%m/%Y") >= ?', [$date])
        ->first();
        

        if (!$semester) {
            
            throw ValidationException::withMessages([
                'date' => trans('validation.The selected date does not fall within any valid semester.')
            ]);
        }

        // If semester is found, define the validation rules
        $start_date = \Carbon\Carbon::createFromFormat('d/m/Y', $semester->start_date)->format('Y-m-d');
        $end_date = \Carbon\Carbon::createFromFormat('d/m/Y', $semester->end_date)->format('Y-m-d');

        return [
            'date_type' => 'required|in:hijri,coptic',
            'date' => [
                "required",
                "after_or_equal:$start_date",
                "before_or_equal:$end_date",
            ],
        ];
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation()
    {
        $date = $this->date;
        if (isset($this->date_type) && isset($this->date)) {
            if ($this->date_type == "hijri") {
                $date = Helper::getCoptic($this->date);
            }
        }

        $this->merge([
            'date' => \Carbon\Carbon::createFromFormat('d/m/Y', $date)->format('Y-m-d'),
        ]);
    }
}
