<?php
namespace App\Http\Controllers\Api\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;
use Str;
use Hash;
use App\Models\User;

use App\Traits\SmsTrait;

use App\Http\Resources\User\UserResource;

class AuthController extends Controller
{
    use SmsTrait;

    /**
     * function that validate token before login
     */
    public function validateToken(Request $request){
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json(['message' => trans('api.Token not provided')], 401);
        }
        
        try {
            // Validate the token
            JWTAuth::setToken($token)->checkOrFail();
            $user = JWTAuth::setToken($token)->authenticate();
        } catch (JWTException $e) {
            return response()->json(['message' => trans('api.Invalid token')], 401);
        }
        return response()->json(['user' => new UserResource($user),'message'=>trans('api.user data')]);
    }

    /* 
        Login function
        user can login with email or phone number 
    */
    public function authenticate(Request $request){

        if(is_numeric($request->get('email'))){
            $credentials= ['phone'=>'966'.$request->get('email'),'password'=>$request->get('password')];
        }
        elseif (filter_var($request->get('email'), FILTER_VALIDATE_EMAIL)) {
            $credentials= ['email' => $request->get('email'), 'password'=>$request->get('password')];
        }

        try {
            if (! $token = JWTAuth::attempt($credentials)) {
                return response()->json(['message'=>trans('api.invalid_credentials')],401);
            }
        } catch (JWTException $e) {
            return response()->json(['message' => trans('api.could_not_create_token')],500);
        }
        $user = JWTAuth::user($token);
        $user->verification_code = mt_rand(100000, 999999);
        $user ->save();
        
        if($user->status == 'active'){

            /** send otp */
            $this->sendOTP($user->phone,$user->verification_code);
    
            $user = new UserResource ($user);
            return response()->json(['otp'=>$user->verification_code,'phone'=>$user->phone,'message'=>trans('api.verfiy login via otp login verfication endpoint')],200);
        }else{
             /** send otp */
             $this->sendOTP($user->phone,$user->verification_code);
            return response()->json(['otp'=>$user->verification_code,'phone'=>$user->phone, 'verification_code'=>$user->verification_code, 'message'=>trans('api.account no verified yet')],403);
        }
    }
    
    /* 
        function to verfiy login account user
    */
    public function verfiyLogin(Request $request){        
        
        $user =  User::where('phone',$request->phone)->where('verification_code',$request->otp)->firstOrFail();
        if($user){
            $user->verification_code = mt_rand(100000, 999999);
            $user->save();

            $token = JWTAuth::fromUser($user);

            $user = new UserResource ($user);
            return response()->json(['userData'=>$user,'token'=>$token,'message'=>trans('api.login verfied successfully')]);
        }else{
            return response()->json(['message'=>trans('api.otp not valid')],403);
        }
   
    }
    
    /* 
        refresh token function to verfiy account user
    */
    public function refreshToken(Request $request){        
        try{
            $user = User::where('email',$request->email)->orwhere('phone',$request->email)->firstOrFail();
            if($user) {
                $token = JWTAuth::fromUser($user);
                return response()->json(compact('token'));
            }else{
                return response()->json(['error' => 'invalid_credentials'],401);
            }
        }catch(JWTException $e) {
            return response()->json(['error' => 'could_not_create_token'],500);
        }       
    }

    /* registeration function */
    public function register(Request $request){
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',    
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'required|string|unique:users',
            'password' => 'required|string|min:8|confirmed',
            // 'remember_token'=>'required|string',
        ]);

        if($validator->fails()){
           //return response()->json($validator->errors(), 400);

            $errors = $validator->errors();
            foreach ($errors->all() as $message) {
                $error[] =  trans("api.$message");
            }
            return response()->json(['message'=>trans('api.error entering data'),'errors'=>$validator->errors()],400);
        }
        
        $user = new User();
        $user->name = $request->name ;
        $user->email = $request->email ;
        $user->phone = '966'.$request->phone ;
        $user->password = Hash::make($request->password) ;
        // $user->remember_token = $request->remember_token ;
        $user->status = 'inactive' ;
        $user->user_type = 'school_manager' ;
        $user->verification_code = mt_rand(100000, 999999);
        $user->save();
 
        $token = JWTAuth::fromUser($user);
        
        /** send otp */
        $this->sendOTP($user->phone,$user->verification_code);

        $user = new UserResource ($user);        
        
        return response()->json(['otp'=>$user->verification_code,'userData'=>$user,'message'=>trans('api.new user registered verfication code will be sent in a few seconds')]);
    }


    public function accountVerfication(Request $request){        
        
        $user =  User::where('phone',$request->phone)->where('verification_code',$request->code)->firstOrFail();
        if($user){
            $user->status = 'active' ;
            $user->verification_code = mt_rand(100000, 999999);
            $user->save();

            /** send otp */
            $this->sendOTP($user->phone,$user->verification_code);

            $token = JWTAuth::fromUser($user);

            $user = new UserResource ($user);
            return response()->json(['userData'=>$user,'token'=>$token,'message'=>trans('api.account verfied successfully')]);
        }else{
            return response()->json(['message'=>trans('api.phone and verfication code not valid')],403);
        }
   
    }

    public function getUserProfile(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }

        $token = JWTAuth::fromUser($user);
        $user = new UserResource ($user);
        return response()->json(['userData'=>$user,'token'=>$token,'message'=>trans('api.user data')]);
    }
    
    /////////// function update user profile ///////////////
    public function updateUserProfile(Request $request){
        $token = $request->bearerToken();
        if($token){
            $user = JWTAuth::toUser($token);
        }

        $user->name=$request->name;
        $user->email=$request->email;
        $user->phone=$request->phone;
        $user->remember_token = $request->remember_token;  
        $user->save();

        $token = JWTAuth::fromUser($user);
        $user = new UserResource ($user);

        return response()->json(['userData'=>$user,'token'=>$token,'message'=>trans('api.user data updated successfully')]);
    }


    /////////////// public function send reset email ////////
    public function sendResetPasswordEmail(Request $request){
        try{
            $tempPassword = mt_rand(10000000, 99999999);
            $user = User::where('email', $request->email)->firstOrFail();
            $user->password = Hash::make($tempPassword);
            $user->save();
            
            ////// send reset password email////////
            $data = ['tempPassword'=>$tempPassword];
            // Mail::send('emails/reset-pasword', $data, function($msg) use ($user) {
            //     $msg->to($user->email, 'Laam App')->subject('reset pasword');
            //     $msg->from(config('mail.from.address'),config('mail.from.name'));
            // });
            
            return response()->json(['tempPassword'=>$tempPassword,'message'=>trans('api.reset password email sent successfully')]);
        }catch(\Exception $e) {
            return response()->json(['message'=>trans('api.email not found')],422);
        }
    }

    /////////////// public function send reset sms ////////
    public function sendResetPasswordViaSms(Request $request){
        try{
            $tempPassword = mt_rand(10000000, 99999999);
            $user = User::where('phone', $request->phone)->firstOrFail();
            $user->password = Hash::make($tempPassword);
            $user->save();
            
            ////// send reset password sms////////
            $this->sendResetPassword($user->phone,$tempPassword);

            $user->verification_code = mt_rand(100000, 999999);
            $user->save();
            
            
            return response()->json(['tempPassword'=>$tempPassword,'message'=>trans('api.reset password sms sent successfully')]);
        }catch(\Exception $e) {
            return response()->json(['message'=>trans('api.no user found with this phone number')],422);
        }
    }

    /** function that remove user data */
    public function removeUser(request $request){
        $user = User::where('phone',$request->phone)->firstOrFail();

        if ($user) {
           $user->delete();
           return response()->json(['message'=>trans('api.user deleted successfully')]);

        }else{
            return response()->json(['message'=>trans('api.no user found with this number')],404);
        }            
    }

    /** function that remove user data */
    public function resndOtp(request $request){
        $user =  User::where('phone',$request->phone)->firstOrFail();
        $user->verification_code = mt_rand(100000, 999999);
        $user ->save();

        /** send otp */
        $this->sendOTP($user->phone,$user->verification_code);
        return response()->json(['verification_code'=>$user->verification_code,'message'=>trans('api.otp sent successfully')]);
          
    }

}