<?php

namespace App\Http\Controllers\Api\School;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\School\SchoolSemesterVacationStoreRequest;
use App\Http\Requests\School\SchoolSemesterVacationUpdateRequest;
use App\Http\Resources\School\SchoolSemesterVacationResource;
use App\Models\AdminDashboard\School\SchoolSemester;
use App\Models\AdminDashboard\School\SchoolSemesterVacation;

class SchoolSemesterVacationController extends Controller
{
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct(){
        //$this->middleware(['auth:api','permission:school-semester-vacations']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request){
        // Start with the base query
        $data = SchoolSemesterVacation::query();

        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            return !in_array($key,['date','date_type','per_page']);
        }, ARRAY_FILTER_USE_KEY);

        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\AdminDashboard\School\SchoolSemesterVacation', $filterData);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        if (isset($queryParams['date_type']) && isset($queryParams['date'])) {
            $date = $queryParams['date'];
            if ($queryParams['date_type'] == "hijri") {
                $date = Helper::getCoptic($queryParams['date']);
            }
            $data = $data->where('date', 'like', "%$date%");
        }

        $filterData = SchoolSemesterVacationResource::collection($data->paginate($perPage));

        return response()->json([
            'semesterVacations' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored 
        by using ID
    */
    public function getById(SchoolSemesterVacation $semesterVacation){
        $semesterVacation = new SchoolSemesterVacationResource($semesterVacation);
        return response()->json(['semesterVacation'=>$semesterVacation,'message'=>trans('api.semester vacation details')],200);
    }

    /* this method will stord data to DB Model*/
    public function store(SchoolSemesterVacationStoreRequest $request){
        $lang = $request->header('lang');
        $semester = SchoolSemester::whereRaw('STR_TO_DATE(start_date, "%d/%m/%Y") <= ?', [$request->date])
                            ->whereRaw('STR_TO_DATE(end_date, "%d/%m/%Y") >= ?', [$request->date])
                            ->first();
        $semesterVacation = new SchoolSemesterVacation();
        $semesterVacation->semester_id = $semester->id;
        $semesterVacation->date = \Carbon\Carbon::createFromFormat('Y-m-d', $request->date)->format('d/m/Y');
        $semesterVacation->date_type = 'coptic';
        $semesterVacation->status = $request->status;
        
        \Carbon\Carbon::setLocale($lang);
        $date = \Carbon\Carbon::parse($request->date);
        $semesterVacation->day = $date->dayName;
        
        $start_date = \Carbon\Carbon::createFromFormat('d/m/Y',$semester->start_date)->subDay();
        $end_date = \Carbon\Carbon::createFromFormat('d/m/Y',$semester->end_date)->addDay();
        if ($date->between($start_date, $end_date)) {
            // Calculate the number of weeks from the start date to the specific date
            $weekOrder = $start_date->diffInWeeks($date) + 1;
            $semesterVacation->week_number = Helper::numberToWord($weekOrder,$lang);
        }
            $semesterVacation->save();

        return response()->json(['message'=>trans('api.record stored successflly')],200);
    }

    /* this method will update specific Model recored 
        by using ID
    */
    public function update(SchoolSemesterVacationUpdateRequest $request,SchoolSemesterVacation $semesterVacation){
        $lang = $request->header('lang');
        $semester = SchoolSemester::whereRaw('STR_TO_DATE(start_date, "%d/%m/%Y") <= ?', [$request->date])
                            ->whereRaw('STR_TO_DATE(end_date, "%d/%m/%Y") >= ?', [$request->date])
                            ->first();
        $semesterVacation->date = \Carbon\Carbon::createFromFormat('Y-m-d', $request->date)->format('d/m/Y');
        $semesterVacation->date_type = 'coptic';
        $semesterVacation->status = $request->status;
        
        \Carbon\Carbon::setLocale($lang);
        $date = \Carbon\Carbon::parse($request->date);
        $semesterVacation->day = $date->dayName;

        $start_date = \Carbon\Carbon::createFromFormat('d/m/Y',$semester->start_date)->subDay();
        $end_date = \Carbon\Carbon::createFromFormat('d/m/Y',$semester->end_date)->addDay();

        if ($date->between($start_date, $end_date)) {
            // Calculate the number of weeks from the start date to the specific date
            $weekOrder = $start_date->diffInWeeks($date) + 1;
            $semesterVacation->week_number = Helper::numberToWord($weekOrder,$lang);
        }
        $semesterVacation->save();

        return response()->json(['message'=>trans('api.record updated successflly')],200);
    }

    /* this method will change activation status for
       specific Model recored by using ID
    */
    public function changeStatus(Request $request,SchoolSemesterVacation $semesterVacation){
        $semesterVacation->status = $request->status;
        $semesterVacation->save();

        return response()->json(['message'=>trans('api.status changed successflly')],200);
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive(SchoolSemesterVacation $semesterVacation){
        $semesterVacation->delete();
        return response()->json(['message'=>trans('api.record archived successflly')],200);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function destroy(SchoolSemesterVacation $semesterVacation){
        $semesterVacation->forceDelete();
        return response()->json(['message'=>trans('api.record deleted successflly')],200);
    }

    /* this method will restore specific Model recored 
        by using ID
    */
    public function restore(SchoolSemesterVacation $semesterVacation){
        $semesterVacation->restore();
        return response()->json(['message'=>trans('api.record restored successflly')],200);
    }

    
}