<?php

namespace App\Http\Controllers\Api\ManagerDashboard\TeamAndCommitteeMeetings;

use App\Helpers\Helper;
use App\Http\Controllers\Controller;
use App\Http\Requests\ManagerDashboard\TeamAndCommitteeMeetings\MeetingDetailsRequest;
use App\Http\Requests\ManagerDashboard\TeamAndCommitteeMeetings\MeetingRequest;
use App\Http\Requests\ManagerDashboard\TeamAndCommitteeMeetings\MeetingUpdateRequest;
use Illuminate\Http\Request;
use App\Http\Resources\ManagerDashboard\TeamAndCommitteeMeetings\MeetingResource;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Meeting;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\MeetingRecommendation;
use App\Models\ManagerDashboard\TeamAndCommitteeMeetings\MeetingSchedual;
use App\Traits\LogsActivity;

class CommitteeMeetingController extends Controller
{
    use LogsActivity;
    /* to uses the class methods
        -you should be authenticated 
        -you should be authorized and have this permission
    */
    public function __construct(){
        //$this->middleware(['auth:api','permission:school-semester-vacations']);
    }

    /*
        -this method will return all Model recoreds
    */
    public function getAll(Request $request){
        // Start with the base query
        $data = Meeting::query();

        // Retrieve all query parameters
        $queryParams = $request->query();
        $filterData = array_filter($queryParams, function ($key) {
            // return $key !== 'date';
            return !in_array($key,['date','date_type','committee_id','created_at','per_page']);
        }, ARRAY_FILTER_USE_KEY);
        
        // Apply filters if any
        if (!empty($filterData)) {
            $data = Helper::filterData('App\Models\ManagerDashboard\TeamAndCommitteeMeetings\Meeting', $filterData);
        }

        // Paginate results with a default value if not provided
        $perPage = $queryParams['per_page'] ?? 15;

        if (isset($queryParams['date_type']) && isset($queryParams['date'])) {
            $date = $queryParams['date'];
            if ($queryParams['date_type'] == "hijri") {
                $date = Helper::getCoptic($queryParams['date']);
            }
            $data = $data->where('date', 'like', "%$date%");
        }

        if (isset($queryParams['date_type']) && isset($queryParams['created_at'])) {
            $date = $queryParams['created_at'];

            if ($queryParams['date_type'] == "hijri") {
                $date = Helper::getCoptic($queryParams['created_at']);
            }
            $date = \Carbon\Carbon::parse($date)->format('Y-d-m');
            $data = $data->whereDate('created_at',$date);
        }

        if($request->committee_id) {
            $filterData = MeetingResource::collection($data->where('model_type','committees')->where('model_id',$queryParams['committee_id'])->paginate($perPage));
        } else {
            $filterData = MeetingResource::collection($data->where('model_type','committees')->paginate($perPage));
        }

        return response()->json([
            'meetings' => $filterData,
            'message' => trans('api.list of all available records'),
        ]);
    }

    /* this method will return specific Model recored 
        by using ID
    */
    public function getById(Meeting $meeting){
        $meeting = new MeetingResource($meeting);
        return response()->json(['meeting'=>$meeting,'message'=>trans('api.meeting details')],200);
    }

    /* this method will stord data to DB Model*/
    public function store(MeetingRequest $request){
        $lang = $request->header('lang');
        $date = \Carbon\Carbon::createFromFormat('Y-m-d',$request->date)->format('d/m/Y');

        $meeting = new Meeting();
        $meeting->name = $request->name;
        $meeting->model_type = 'committees';
        $meeting->model_id = $request->model_id;
        $meeting->school_id = $request->school_id;
        $meeting->semester_id = $request->semester_id;
        $meeting->meeting_type = $request->meeting_type;
        $meeting->date = $date;
        $meeting->date_type = 'coptic';
        $meeting->start_time = $request->start_time;
        $meeting->place = $request->place;
        $meeting->meeting_status = 'not_completed';
        $meeting->save();

        return response()->json([
            'meeting_id' => $meeting->id,
            'date' => $meeting->date,
            'message' => trans('api.record stored successflly'),
        ]);
    }

    /* this method will update specific Model recored 
        by using ID
    */
    public function storeExtra(MeetingDetailsRequest $request, Meeting $meeting)
    {
        $lang = $request->header('lang');
        $meeting->attendances_count = $request->attendances_count;
        $meeting->target_audience = $request->target_audience;
        $meeting->end_time = $request->end_time;
        $meeting->meeting_status = 'completed';
        $meeting->save();

        if ($request->schedual_details) {
            foreach ($request->schedual_details as $detail) {
                MeetingSchedual::create([
                    'meeting_id' => $meeting->id,
                    'details' => $detail['details'],
                ]);
            }
        }

        if ($request->recommendations) {
            foreach ($request->recommendations as $recommendation) {
                MeetingRecommendation::create([
                    'meeting_id' => $meeting->id,
                    'recommendation' => $recommendation['recommendation'],
                    'executive_authority' => $recommendation['executive_authority'],
                    'execution_time' => $recommendation['execution_time'],
                    'belonged_executive_authority' => $recommendation['belonged_executive_authority'],
                    'recommendation_status' => $recommendation['recommendation_status'],
                    'reason_of_not_execution' => $recommendation['reason_of_not_execution'],
                ]);
            }
        }
        // ------------------------------- School ID -----------------------
        $school_id = $meeting->school_id;
        $this->ManagerlogActivity('Meeting', 'store', $school_id, '/committees-meetings/' . $meeting->id . '/details');


        return response()->json([
            'date' => $meeting->date,
            'message' => trans('api.record updated successfully'),
        ]);
    }


    /* this method will update specific Model recored 
        by using ID
    */
    public function update(MeetingUpdateRequest $request, $meeting_id)
    {
        $lang = $request->header('lang');

        $date = \Carbon\Carbon::createFromFormat('Y-m-d',$request->date)->format('d/m/Y');
        $meeting = Meeting::where('id',$meeting_id)->where('model_type','committees')->firstOrFail();
        $meeting->name = $request->name;
        $meeting->model_id = $request->model_id;
        $meeting->school_id = $request->school_id;
        $meeting->semester_id = $request->semester_id;
        $meeting->meeting_type = $request->meeting_type;
        $meeting->date = $date;
        $meeting->start_time = $request->start_time;
        $meeting->place = $request->place;
        $meeting->attendances_count = $request->attendances_count;
        $meeting->target_audience = $request->target_audience;
        $meeting->end_time = $request->end_time;
        $meeting->save();
    
        MeetingSchedual::where('meeting_id', $meeting->id)->forceDelete();
        if (isset($request->schedual_details)) {
            foreach ($request->schedual_details as $detail) {
                MeetingSchedual::create([
                    'meeting_id' => $meeting->id,
                    'details' => $detail['details'],
                ]);
            }
        }
    
        MeetingRecommendation::where('meeting_id', $meeting->id)->forceDelete();
        if (isset($request->recommendations)) {
            foreach ($request->recommendations as $recommendation) {
                MeetingRecommendation::create([
                    'meeting_id' => $meeting->id,
                    'recommendation' => $recommendation['recommendation'],
                    'executive_authority' => $recommendation['executive_authority'],
                    'execution_time' => $recommendation['execution_time'],
                    'belonged_executive_authority' => $recommendation['belonged_executive_authority'],
                    'recommendation_status' => $recommendation['recommendation_status'],
                    'reason_of_not_execution' => $recommendation['reason_of_not_execution'],
                ]);
            }
        }

        // ------------------------------- School ID -----------------------
        $school_id = $meeting->school_id;
        $this->ManagerlogActivity('Meeting', 'update', $school_id, '/committees-meetings/' . $meeting->id . '/details');

        return response()->json([
            'date' => $meeting->date,
            'message' => trans('api.record updated successfully'),
        ]);
    }

    /* this method will change activation status for
       specific Model recored by using ID
    */
    public function saveAsDraft(MeetingDetailsRequest $request,Meeting $meeting){
        $lang = $request->header('lang');
        $meeting->attendances_count = $request->attendances_count;
        $meeting->target_audience = $request->target_audience;
        $meeting->end_time = $request->end_time;
        $meeting->meeting_status = 'drafted';
        $meeting->save();

        if ($request->schedual_details) {
            foreach ($request->schedual_details as $detail) {
                MeetingSchedual::create([
                    'meeting_id' => $meeting->id,
                    'details' => $detail['details'],
                ]);
            }
        }

        if ($request->recommendations) {
            foreach ($request->recommendations as $recommendation) {
                MeetingRecommendation::create([
                    'meeting_id' => $meeting->id,
                    'recommendation' => $recommendation['recommendation'],
                    'executive_authority' => $recommendation['executive_authority'],
                    'execution_time' => $recommendation['execution_time'],
                    'belonged_executive_authority' => $recommendation['belonged_executive_authority'],
                    'recommendation_status' => $recommendation['recommendation_status'],
                    'reason_of_not_execution' => $recommendation['reason_of_not_execution'],
                ]);
            }
        }

        return response()->json(['message' => trans('api.record saved as draft successfully')], 200);
    }

     /* this method will update status of specific Model recored 
        by using ID
    */
    public function changeStatus(Request $request,Meeting $meeting){
        $meeting->meeting_status = $request->status;
        $meeting->save();
        return response()->json(['message'=>trans('api.status changed successfully')],200);
    }

    /* this method will archive specific Model recored 
        by using ID
    */
    public function archive(Meeting $meeting){
        $school_id = 1; //$meeting->school_id;
        $meeting->delete();
        $this->ManagerlogActivity('MeetingSchedual', 'archive', $school_id);

        return response()->json(['message'=>trans('api.record archived successfully')],200);
    }

    /* this method will delete specific Model recored 
        by using ID
    */
    public function destroy(Meeting $meeting){
        $school_id = 1; //$meeting->school_id;
        $meeting->forceDelete();
        $this->ManagerlogActivity('MeetingSchedual', 'delete', $school_id);
        return response()->json(['message'=>trans('api.record deleted successfully')],200);
    }

    /* this method will restore specific Model recored 
        by using ID
    */
    public function restore(Meeting $meeting){
        $school_id = 1; //$meeting->school_id;
        $meeting->restore();
        $this->ManagerlogActivity('MeetingSchedual', 'restore', $school_id);
        return response()->json(['message'=>trans('api.record restored successfully')],200);
    }

    
}